/*******************************************************************************
* This software program is available to you under a choice of one of two 
* licenses. You may choose to be licensed under either the GNU General Public 
* License 2.0, June 1991, available at http://www.fsf.org/copyleft/gpl.html, 
* or the Intel BSD + Patent License, the text of which follows:
* 
* Recipient has requested a license and Intel Corporation ("Intel") is willing
* to grant a license for the software entitled Linux Intel Advanced Network
* Services (iANS) (the "Licensed Program") being provided by Intel Corporation.
* The following definitions apply to this license:
* 
* "Licensed Patents" means patent claims licensable by Intel Corporation which 
* are necessarily infringed by the use of sale of the Software alone or when 
* combined with the operating system referred to below.
* 
* "Recipient" means the party to whom Intel delivers this Software.
* 
* "Licensee" means Recipient and those third parties that receive a license to 
* any operating system available under the GNU General Public License 2.0 or 
* later.
* 
* Copyright (c) 1999 - 2002 Intel Corporation.
* All rights reserved.
* 
* The license is provided to Recipient and Recipient's Licensees under the 
* following terms.
* 
* Redistribution and use in source and binary forms of the Software, with or 
* without modification, are permitted provided that the following conditions 
* are met:
* 
* Redistributions of source code of the Software may retain the above 
* copyright notice, this list of conditions and the following disclaimer.
* 
* Redistributions in binary form of the Software may reproduce the above 
* copyright notice, this list of conditions and the following disclaimer in 
* the documentation and/or materials provided with the distribution.
* 
* Neither the name of Intel Corporation nor the names of its contributors 
* shall be used to endorse or promote products derived from this Software 
* without specific prior written permission.
* 
* Intel hereby grants Recipient and Licensees a non-exclusive, worldwide, 
* royalty-free patent license under Licensed Patents to make, use, sell, offer 
* to sell, import and otherwise transfer the Software, if any, in source code 
* and object code form. This license shall include changes to the Software 
* that are error corrections or other minor changes to the Software that do 
* not add functionality or features when the Software is incorporated in any 
* version of an operating system that has been distributed under the GNU 
* General Public License 2.0 or later. This patent license shall apply to the 
* combination of the Software and any operating system licensed under the GNU 
* General Public License 2.0 or later if, at the time Intel provides the 
* Software to Recipient, such addition of the Software to the then publicly 
* available versions of such operating systems available under the GNU General 
* Public License 2.0 or later (whether in gold, beta or alpha form) causes 
* such combination to be covered by the Licensed Patents. The patent license 
* shall not apply to any other combinations which include the Software. NO 
* hardware per se is licensed hereunder.
* 
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
* AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
* IMPLIED WARRANTIES OF MECHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
* ARE DISCLAIMED. IN NO EVENT SHALL INTEL OR IT CONTRIBUTORS BE LIABLE FOR ANY 
* DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
* (INCLUDING, BUT NOT LIMITED, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
* ANY LOSS OF USE; DATA, OR PROFITS; OR BUSINESS INTERUPTION) HOWEVER CAUSED 
* AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY OR 
* TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/

/*****************************************************************************
******************************************************************************
**                                                                          **
** INTEL CORPORATION                                                        **
**                                                                          **
** This software is supplied under the terms of the license included        **
** above.  All use of this software must be in accordance with the terms    **
** of that license.                                                         **
**                                                                          **
** THIS FILE IS USED WHEN CREATING ians_core.o. HENCE, IT SHOULD NOT BE     **
** MODIFIED!                                                                **
**                                                                          **
**  Module Name:    incg_dev.c                                              **
**                                                                          **
**  Abstract:       This module contains all the functions needed for the   **
**                  interface of network devices with the kernel.           **
**                                                                          **
**  Environment:    Kernel Mode (linux 2.2.x, 2.4.x)                        **
**                                                                          **
**                  Written using Tab Size = 4, Indent Size = 4             **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
**      01-01-2002  Shmulik Hen,    1st created                             **
**                                                                          **
******************************************************************************
*****************************************************************************/

#include <linux/netdevice.h>
#include <linux/if.h>
#include <linux/rtnetlink.h>

#include "ians_kcompat.h"
#include "ians_status.h"
#include "ans_interface.h"
#include "lib/incg_open_utils.h"
#include "lib/incg_defs.h"
#include "lib/incg_gp_mem.h"
#include "lib/incg_types.h"


/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
IANS_STATUS iansRegisterDev(device_t *ansDev)
{
    struct net_device * dev;
    BOOLEAN ioctlContext = ANSGetConfigExecuteCommand();

    ASSERT(ansDev);
    ASSERT(ansDev->name);

    if(!ioctlContext) {
        IANS_PRINT_ERROR("Not in ioctl context!!!\n");
    }

    dev = (struct net_device *) GET_OS_DEV(ansDev);

    if(dev == NULL) {
        return IANS_VADAPTER_REGISTER_FAIL;
    }

    //IANS_PRINT_DEBUG("Registering vadapter %s\n", dev->name);

    register_netdevice(dev);

    return IANS_OK;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
void iansUnregisterDev(device_t *ansDev, BOOLEAN close)
{
    struct net_device * devExist = NULL;
    BOOLEAN ioctlContext = ANSGetConfigExecuteCommand();

    ASSERT(ansDev);
    ASSERT(ansDev->name);

    if((devExist = dev_get_by_name(ansDev->name))) {
#ifdef LINUX_KERNEL_24X
        dev_put(devExist);
#endif //LINUX_KERNEL_24X
        if(devExist != (struct net_device *) GET_OS_DEV(ansDev)) {
            IANS_PRINT_ERROR("Mismatch of kernel and ANS pointers to %s's device structure - cann't unregister the net device!\n", ansDev->name);
            return;
        }
        else {//unregister the device.
            if(close) {
                dev_close(devExist);
            }

            (ioctlContext ? unregister_netdevice(devExist) : unregister_netdev(devExist));
        }
    }
    else {
        IANS_PRINT_ERROR("Device %s doesn't exist - cann't unregister the net device!\n", ansDev->name);
    }
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
IANS_STATUS iansSetMaster(device_t *ansSlave, device_t *ansMaster)
{
#ifdef LINUX_KERNEL_24X
    struct net_device *slave = (struct net_device *)GET_OS_DEV(ansSlave);
    struct net_device *master = NULL;
    BOOLEAN ioctlContext = ANSGetConfigExecuteCommand();
    IANS_STATUS res = IANS_OK;

    ASSERT(slave);

    if(ansMaster) { //enslave
        master = (struct net_device *)GET_OS_DEV(ansMaster);
    }

    if(!ioctlContext) {
        rtnl_lock();
    }
    if(netdev_set_master(slave, master) != 0) {
        res = (master ? IANS_FAILED_ENSLAVE_MEMBER : IANS_FAILED_RELEASE_MEMBER);
    }

    if(!ioctlContext) {
        rtnl_unlock();
    }

    return res;
#endif //LINUX_KERNEL_24X
#ifdef LINUX_KERNEL_22X
    return IANS_OK;
#endif //LINUX_KERNEL_22X
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
void iansDevPut(device_t *ansDev)
{
#ifdef LINUX_KERNEL_24X
    struct net_device *dev = (struct net_device *) GET_OS_DEV(ansDev);
    ASSERT(dev);

    dev_put(dev);
#endif //LINUX_KERNEL_24X
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
BOOLEAN iansIfStopped(device_t *ansDev)
{
    struct net_device *dev = (struct net_device *) GET_OS_DEV(ansDev);
    ASSERT(dev);

    return netif_queue_stopped(dev);
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
void iansIfStop(device_t *ansDev)
{
    struct net_device *dev = (struct net_device *) GET_OS_DEV(ansDev);
    ASSERT(dev);

    netif_stop_queue(dev);
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
void iansIfWake(device_t *ansDev)
{
    struct net_device *dev = (struct net_device *) GET_OS_DEV(ansDev);
    ASSERT(dev);

    netif_wake_queue(dev);
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
BOOLEAN iansBaseDevIsOpen(device_t *ansDev)
{
    struct net_device *dev=NULL;
    ASSERT(ansDev);
    dev = (struct net_device *) GET_OS_DEV(ansDev);
    ASSERT(dev);

    if(netif_running(dev) || (dev->flags & IFF_UP)) {
        return TRUE;
    }
    return FALSE;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
ULONG iansGetDeviceBaseAddr(device_t *ansDev)
{
    struct net_device *dev=NULL;
    ASSERT(ansDev);
    dev = (struct net_device *) GET_OS_DEV(ansDev);
    ASSERT(dev);

    return(ULONG) dev->base_addr;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
IANS_STATUS iansGetDevBaseAddrByName(char *devName, UINT32 *baseAddr)
{
    struct net_device *dev;

    dev = dev_get_by_name(devName);
    if(dev == NULL) {
        //IANS_PRINT_ERROR("Device %s doesn't exist!\n", devName);
        return IANS_ERROR;
    }

    *baseAddr = (UINT32)dev->base_addr;

#ifdef LINUX_KERNEL_24X
    dev_put(dev);
#endif //LINUX_KERNEL_24X

    return IANS_OK;
}

