/*******************************************************************************
* This software program is available to you under a choice of one of two 
* licenses. You may choose to be licensed under either the GNU General Public 
* License 2.0, June 1991, available at http://www.fsf.org/copyleft/gpl.html, 
* or the Intel BSD + Patent License, the text of which follows:
* 
* Recipient has requested a license and Intel Corporation ("Intel") is willing
* to grant a license for the software entitled Linux Intel Advanced Network
* Services (iANS) (the "Licensed Program") being provided by Intel Corporation.
* The following definitions apply to this license:
* 
* "Licensed Patents" means patent claims licensable by Intel Corporation which 
* are necessarily infringed by the use of sale of the Software alone or when 
* combined with the operating system referred to below.
* 
* "Recipient" means the party to whom Intel delivers this Software.
* 
* "Licensee" means Recipient and those third parties that receive a license to 
* any operating system available under the GNU General Public License 2.0 or 
* later.
* 
* Copyright (c) 1999 - 2002 Intel Corporation.
* All rights reserved.
* 
* The license is provided to Recipient and Recipient's Licensees under the 
* following terms.
* 
* Redistribution and use in source and binary forms of the Software, with or 
* without modification, are permitted provided that the following conditions 
* are met:
* 
* Redistributions of source code of the Software may retain the above 
* copyright notice, this list of conditions and the following disclaimer.
* 
* Redistributions in binary form of the Software may reproduce the above 
* copyright notice, this list of conditions and the following disclaimer in 
* the documentation and/or materials provided with the distribution.
* 
* Neither the name of Intel Corporation nor the names of its contributors 
* shall be used to endorse or promote products derived from this Software 
* without specific prior written permission.
* 
* Intel hereby grants Recipient and Licensees a non-exclusive, worldwide, 
* royalty-free patent license under Licensed Patents to make, use, sell, offer 
* to sell, import and otherwise transfer the Software, if any, in source code 
* and object code form. This license shall include changes to the Software 
* that are error corrections or other minor changes to the Software that do 
* not add functionality or features when the Software is incorporated in any 
* version of an operating system that has been distributed under the GNU 
* General Public License 2.0 or later. This patent license shall apply to the 
* combination of the Software and any operating system licensed under the GNU 
* General Public License 2.0 or later if, at the time Intel provides the 
* Software to Recipient, such addition of the Software to the then publicly 
* available versions of such operating systems available under the GNU General 
* Public License 2.0 or later (whether in gold, beta or alpha form) causes 
* such combination to be covered by the Licensed Patents. The patent license 
* shall not apply to any other combinations which include the Software. NO 
* hardware per se is licensed hereunder.
* 
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
* AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
* IMPLIED WARRANTIES OF MECHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
* ARE DISCLAIMED. IN NO EVENT SHALL INTEL OR IT CONTRIBUTORS BE LIABLE FOR ANY 
* DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
* (INCLUDING, BUT NOT LIMITED, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
* ANY LOSS OF USE; DATA, OR PROFITS; OR BUSINESS INTERUPTION) HOWEVER CAUSED 
* AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY OR 
* TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/

/*****************************************************************************
******************************************************************************
**                                                                          **
** INTEL CORPORATION                                                        **
**                                                                          **
** This software is supplied under the terms of the license included        **
** above.  All use of this software must be in accordance with the terms    **
** of that license.                                                         **
**                                                                          **
** THIS FILE IS USED WHEN CREATING ians_core.o. HENCE, IT SHOULD NOT BE     **
** MODIFIED!                                                                **
**                                                                          **
**  Module Name:    ians_base.c                                             **
**                                                                          **
**  Abstract:       This module contains all the functions needed for the   **
**                  interface of the iANS module with the base driver.      **
**                                                                          **
**  Environment:    Kernel Mode (linux 2.2.x, 2.4.x)                        **
**                                                                          **
**                  Written using Tab Size = 4, Indent Size = 4             **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
******************************************************************************
*****************************************************************************/


#include <linux/netdevice.h>
#include <linux/rtnetlink.h>

#include "ians_kcompat.h"
#include "ians_status.h"
#include "ans_interface.h"
#include "lib/incg_open_utils.h"
#include "lib/incg_gp_mem.h"
#include "lib/incg_net.h"
#include "lib/incg_types.h"
#include "lib/incg_locks.h"
#include "base_comm_os.h"

#ifdef LINUX_KERNEL_24X
#include <linux/ethtool.h>
#endif //LINUX_KERNEL_24X

#ifndef SIOCGMIIPHY
#define SIOCGMIIPHY (SIOCDEVPRIVATE)        /* Get the PHY in use. */
#define SIOCGMIIREG (SIOCDEVPRIVATE+1)      /* Read a PHY register. */
#endif //SIOCGMIIPHY


/*****************************************************************************
**                  F o r w a r d  D e c l e r a t i o n s                  **
*****************************************************************************/
static void iAnsNotify(struct net_device *dev, int IndType);

/*----------------------- MEMBER DEVICE ENTRY POINTS -----------------------*/
static IANS_STATUS baseOpenDevice(device_t *ansDev);
static IANS_STATUS baseCloseDevice(device_t *ansDev);
static IANS_STATUS baseSend(device_t *ansDev, message_t *msg);
static pstats_t baseGetStats(device_t *ansDev);
static IANS_STATUS baseSetMulticastList(device_t *ansDev, MULTICAST_LIST *multicastList, PACKET_FILTER packetFilter);
static IANS_STATUS baseSetMacAddr(device_t *ansDev, sockAddr_t *addr);
static IANS_STATUS baseDoIoctl(device_t *ansDev, preq_t ifr, int cmd);
static IANS_STATUS baseChangeMtu(device_t *ansDev, int new_mtu);

/*-------------------------- SERVICE FUNCTIONS -----------------------------*/
static IANS_STATUS _addOSMCList(struct dev_mc_list **mcList, int *mcCount);
static void _freeOSMCList(struct dev_mc_list **mcList, int *mcCount);
static inline IANS_STATUS _convertAnsMCListToOsMCList(struct dev_mc_list **osList, int *osCount, MULTICAST_LIST *ansList);
static inline void _convertAnsPFToOsPF(unsigned short *osFlags, PACKET_FILTER packetFilter);
#ifdef LINUX_KERNEL_24X
static void _convertEthToolToStats(struct ethtool_cmd *ecmd, IANS_BD_IOC_PARAM_STATUS *stat);
#endif //LINUX_KERNEL_24X


/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
IANS_STATUS baseInitMemberDev(const char *baseName, device_t **ansDev)
{
    struct net_device *dev;

    if((dev = dev_get_by_name(baseName)) == NULL) {
        IANS_PRINT_ERROR("Device %s doesn't exist!.\n", baseName);
        return  IANS_MEMBER_NAME_WRONG;
    }

    iNCGGPSizedMemAlloc((void **)(ansDev), sizeof(device_t));
    if(*ansDev == NULL) {
        IANS_PRINT_ERROR("Failed to allocate an IANS device structure for a member.\n");
        return IANS_MEM_ERROR;
    }

    (*ansDev)->pOsDev = dev;
    (*ansDev)->name = dev->name;
    (*ansDev)->tbusy = FALSE;
    (*ansDev)->mac = (MAC_ADDR *)(dev->dev_addr);
    (*ansDev)->mtu = &(dev->mtu);
    (*ansDev)->open = baseOpenDevice;
    (*ansDev)->close = baseCloseDevice;
    (*ansDev)->send = (IANS_STATUS (*)(void *, void *)) baseSend;
    (*ansDev)->getStats = baseGetStats;
    (*ansDev)->setMulticastList = baseSetMulticastList;
    (*ansDev)->setMacAddr = baseSetMacAddr;
    (*ansDev)->doIoctl = baseDoIoctl;
    (*ansDev)->changeMtu = baseChangeMtu;

    return IANS_OK;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
IANS_STATUS baseIdentify(device_t *ansDev)
{
    struct net_device *dev = (struct net_device *) GET_OS_DEV(ansDev);
    struct ifreq ifr;
    IANS_BD_PARAM_IDENTIFY id;

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return IANS_IOCTL_FAILED;
    }

    if(dev->do_ioctl == NULL) {
        IANS_PRINT_ERROR("%s: dev->do_ioctl == NULL\n", dev->name);
        return IANS_IOCTL_FAILED;
    }

    id.Header.Opcode = IANS_OP_BD_IDENTIFY;
    memcpy(id.iANSSignature, IntelCopyrightString, IANS_SIGNATURE_LENGTH);
    id.iANSCommVersion = IANS_COMM_VERSION_MINOR;
    id.iANSCommVersion |= (IANS_COMM_VERSION_MAJOR << 16);
    ifr.ifr_data = (void *) &id;

    if((dev->do_ioctl(dev, &ifr, IANS_BASE_SIOC)) != 0) {
        IANS_PRINT_ERROR("%s: did not recognize IANS_OP_BD_IDENTIFY\n", dev->name);
        return IANS_IOCTL_FAILED;
    }
    return IANS_OK;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
IANS_STATUS baseSetExtMode(device_t *ansDev, BOOLEAN attributed_mode)
{
    struct net_device *dev = (struct net_device *) GET_OS_DEV(ansDev);
    struct ifreq ifr;
    IANS_BD_PARAM_EXT_SET_MODE request;

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return IANS_IOCTL_FAILED;
    }

    if(dev->do_ioctl == NULL) {
        IANS_PRINT_ERROR("%s: dev->do_ioctl == NULL\n", dev->name);
        return IANS_IOCTL_FAILED;
    }

    request.Header.Opcode = IANS_OP_EXT_SET_MODE;
    request.BDIansStatusReport = IANS_REQUEST_SUPPORT;
    if(attributed_mode) {
        request.BDIansAttributedMode = IANS_REQUEST_SUPPORT;
    }
    else {
        request.BDIansAttributedMode = IANS_DONT_SUPPORT;
    }
    request.BDIansRoutingMode = IANS_ROUTING_RX_PROTOCOL;

    ifr.ifr_data = (void *) &request;

    if((dev->do_ioctl(dev, &ifr, IANS_BASE_SIOC)) != 0) {
        IANS_PRINT_ERROR("%s: failed IANS_OP_EXT_SET_MODE\n", dev->name);
        return IANS_IOCTL_FAILED;
    }

    return IANS_OK;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
IANS_STATUS baseGetExtendedCapability(device_t *ansDev, IANS_BD_PARAM_EXT_CAP *cap)
{
    struct net_device *dev = (struct net_device *) GET_OS_DEV(ansDev);
    struct ifreq ifr;

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return IANS_IOCTL_FAILED;
    }

    if(dev->do_ioctl == NULL) {
        IANS_PRINT_ERROR("%s: dev->do_ioctl == NULL\n", dev->name);
        return IANS_IOCTL_FAILED;
    }

    cap->Header.Opcode = IANS_OP_EXT_GET_CAPABILITY;

    ifr.ifr_data = (void *) cap;

    if((dev->do_ioctl(dev, &ifr, IANS_BASE_SIOC)) != 0) {
        IANS_PRINT_ERROR("%s: failed IANS_OP_EXT_GET_CAPABILITY\n", dev->name);
        return IANS_IOCTL_FAILED;
    }

    return IANS_OK;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
IANS_STATUS baseGetStatusInfo(device_t *ansDev, IANS_BD_IOC_PARAM_STATUS *stat)
{
    struct net_device *dev = (struct net_device *) GET_OS_DEV(ansDev);
    struct ifreq ifr;

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return IANS_IOCTL_FAILED;
    }
    
    if(dev->do_ioctl == NULL) {
        IANS_PRINT_ERROR("%s: dev->do_ioctl == NULL\n", dev->name);
        return IANS_IOCTL_FAILED;
    }

    stat->Header.Opcode = IANS_OP_EXT_GET_STATUS;

    ifr.ifr_data = (void *) stat;

    if((dev->do_ioctl(dev, &ifr, IANS_BASE_SIOC)) != 0) {
        IANS_PRINT_ERROR("%s: failed IANS_OP_EXT_GET_STATUS\n", dev->name);
        return IANS_IOCTL_FAILED;
    }

    return IANS_OK;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
IANS_STATUS baseGetEthToolStatusInfo(device_t *ansDev, IANS_BD_IOC_PARAM_STATUS *stat)
{
#ifdef LINUX_KERNEL_24X
    struct net_device *dev = (struct net_device *) GET_OS_DEV(ansDev);
    struct ethtool_cmd ecmd;
    struct ifreq ifr;
    char buf[1024];
 
    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return IANS_IOCTL_FAILED;
    }

    if(dev->do_ioctl == NULL) {
        IANS_PRINT_ERROR("%s: dev->do_ioctl == NULL\n", dev->name);
        return IANS_IOCTL_FAILED;
    }

    memset(&ecmd, 0, sizeof(ecmd));
    memset(&ifr, 0, sizeof(ifr));
    ifr.ifr_data = (char*)buf;
    ecmd.cmd = ETHTOOL_GSET;
    memcpy(&buf, &ecmd, sizeof(ecmd.cmd));

    if((dev->do_ioctl(dev, &ifr, SIOCETHTOOL)) != 0) {
        IANS_PRINT_DEBUG("%s: did not recognize SIOCETHTOOL\n", dev->name);
        return IANS_IOCTL_FAILED;
    }

    _convertEthToolToStats((struct ethtool_cmd *) &buf, stat);

    //IANS_PRINT_DEBUG("%s: link status = %d\n", dev->name, (int)stat->Status.LinkStatus);
    //IANS_PRINT_DEBUG("%s: speed = %d\n", dev->name, (int)stat->Status.LinkSpeed);
    //IANS_PRINT_DEBUG("%s: duplex = %d\n", dev->name, (int)stat->Status.Duplex);

#endif //LINUX_KERNEL_24X
    return IANS_OK;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
IANS_STATUS baseMIIGetPhyAddress(device_t *ansDev, UINT16 *phy_addr)
{
    struct net_device *dev = (struct net_device *) GET_OS_DEV(ansDev);
    struct ifreq ifr;
    UINT16 *buf;

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return IANS_IOCTL_FAILED;
    }

    if(dev->do_ioctl == NULL) {
        IANS_PRINT_ERROR("%s: dev->do_ioctl == NULL\n", dev->name);
        return IANS_IOCTL_FAILED;
    }

    memset(&ifr, 0, sizeof(ifr));
    buf = (UINT16*)&(ifr.ifr_data);

    if((dev->do_ioctl(dev, &ifr, SIOCGMIIPHY)) != 0) {
        if(SIOCGMIIPHY == SIOCDEVPRIVATE) {
            IANS_PRINT_DEBUG("%s: did not recognize SIOCGMIIPHY\n", dev->name);
            return IANS_IOCTL_FAILED;
        }

        // quick workaround to bypass the fact that some drivers do not implement the new MII ioctls.
        if((dev->do_ioctl(dev, &ifr, SIOCDEVPRIVATE)) != 0) {
            // the driver has some other bug - can't find any way to read MII.
            IANS_PRINT_DEBUG("%s: did not recognize SIOCGMIIPHY or SIOCDEVPRIVATE\n", dev->name);
            return IANS_IOCTL_FAILED;
        }
    }

    *phy_addr = buf[0];
    return IANS_OK;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
IANS_STATUS baseMIIReadRegister(device_t *ansDev, UINT16 phy_addr, UINT16 reg_addr, UINT16 *reg_data)
{
    struct net_device *dev = (struct net_device *) GET_OS_DEV(ansDev);
    struct ifreq ifr;
    UINT16 *buf;

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return IANS_IOCTL_FAILED;
    }

    if(dev->do_ioctl == NULL) {
        IANS_PRINT_ERROR("%s: dev->do_ioctl == NULL\n", dev->name);
        return IANS_IOCTL_FAILED;
    }

    memset(&ifr, 0, sizeof(ifr));
    buf = (UINT16*)&(ifr.ifr_data);

    buf[0] = phy_addr;
    buf[1] = reg_addr;
    if((dev->do_ioctl(dev, &ifr, SIOCGMIIREG)) != 0) {
        if(SIOCGMIIREG == SIOCDEVPRIVATE+1) {
            IANS_PRINT_DEBUG("%s: did not recognize SIOCGMIIREG\n", dev->name);
            return IANS_IOCTL_FAILED;
        }

        // quick workaround to bypass the fact that some drivers do not implement the new MII ioctls.
        if((dev->do_ioctl(dev, &ifr, SIOCDEVPRIVATE+1)) != 0) {
            IANS_PRINT_DEBUG("%s: did not recognize SIOCGMIIREG or SIOCDEVPRIVATE+1\n", dev->name);
            return IANS_IOCTL_FAILED;
        }
    }

    *reg_data = buf[3];
    return IANS_OK;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
IANS_STATUS baseUndoSettings(device_t *ansDev)
{
    struct net_device *dev = (struct net_device *) GET_OS_DEV(ansDev);
    struct ifreq ifr;
    IANS_BD_PARAM_HEADER request;

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return IANS_IOCTL_FAILED;
    }

    if(dev->do_ioctl == NULL) {
        IANS_PRINT_ERROR("%s: dev->do_ioctl == NULL\n", dev->name);
        return IANS_IOCTL_FAILED;
    }

    request.Opcode = IANS_OP_BD_DISCONNECT;

    ifr.ifr_data = (void *) &request;

    if((dev->do_ioctl(dev, &ifr, IANS_BASE_SIOC)) != 0) {
        IANS_PRINT_ERROR("%s: failed IANS_OP_BD_DISCONNECT\n", dev->name);
        return IANS_IOCTL_FAILED;
    }
    return IANS_OK;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
IANS_STATUS baseSetNotifyCB(device_t *ansDev)
{
    struct net_device *dev = (struct net_device *) GET_OS_DEV(ansDev);
    IANS_BD_ANS_SET_CB request;
    struct ifreq ifr;

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return IANS_IOCTL_FAILED;
    }

    if(dev->do_ioctl == NULL) {
        IANS_PRINT_ERROR("%s: dev->do_ioctl == NULL\n", dev->name);
        return IANS_IOCTL_FAILED;
    }

    request.Header.Opcode = IANS_OP_ANS_SET_CB;
    request.notify = iAnsNotify;

    ifr.ifr_data = (void *) &request;

    if((dev->do_ioctl(dev, &ifr, IANS_BASE_SIOC)) != 0) {
        IANS_PRINT_ERROR("%s: failed IANS_OP_ANS_SET_CB\n", dev->name);
        return IANS_IOCTL_FAILED;
    }
    return IANS_OK;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
MAC_ADDR baseGetMacAddr(device_t *ansDev)
{
    struct net_device *dev = (struct net_device *) GET_OS_DEV(ansDev);
    MAC_ADDR dummyAddr = {DUMMY_MAC_ADDR};

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return dummyAddr;
    }

    return *((MAC_ADDR *)dev->dev_addr);
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
IANS_STATUS baseGetVlanCapability(device_t *ansDev, IANS_BD_PARAM_IVLAN_CAP *cap)
{
    struct net_device *dev = (struct net_device *) GET_OS_DEV(ansDev);
    struct ifreq ifr;

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return IANS_IOCTL_FAILED;
    }

    if(dev->do_ioctl == NULL) {
        IANS_PRINT_ERROR("%s: dev->do_ioctl == NULL\n", dev->name);
        return IANS_IOCTL_FAILED;
    }

    cap->Header.Opcode = IANS_OP_IVLAN_ID_GET_CAPABILITY;

    ifr.ifr_data = (void *) cap;

    if((dev->do_ioctl(dev, &ifr, IANS_BASE_SIOC)) != 0) {
        IANS_PRINT_ERROR("%s: failed IANS_OP_IVLAN_ID_GET_CAPABILITY\n", dev->name);
        return IANS_IOCTL_FAILED;
    }
    return IANS_OK;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
IANS_STATUS baseSetVlanMode(device_t *ansDev, BOOLEAN reqVlanSup, BOOLEAN reqFilterSup)
{
    struct net_device *dev = (struct net_device *) GET_OS_DEV(ansDev);
    struct ifreq ifr;
    IANS_BD_PARAM_IVLAN_SET_MODE request;

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return IANS_IOCTL_FAILED;
    }

    if(dev->do_ioctl == NULL) {
        IANS_PRINT_ERROR("%s: dev->do_ioctl == NULL\n", dev->name);
        return IANS_IOCTL_FAILED;
    }

    request.Header.Opcode = IANS_OP_IVLAN_ID_SET_MODE;

    request.VlanIDRequest = (reqVlanSup ? IANS_REQUEST_SUPPORT : IANS_DONT_SUPPORT);

    request.VlanIDFilteringRequest = (reqFilterSup ? IANS_REQUEST_SUPPORT : IANS_DONT_SUPPORT);

    ifr.ifr_data = (void *) &request;

    if((dev->do_ioctl(dev, &ifr, IANS_BASE_SIOC)) != 0) {
        IANS_PRINT_ERROR("%s: failed IANS_BD_PARAM_IVLAN_SET_MODE\n", dev->name);
        return IANS_IOCTL_FAILED;
    }
    return IANS_OK;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
IANS_STATUS baseSetVlanIdTable(device_t *ansDev, UINT16 *vlanIdTable, UINT32 vlanNum)
{
    struct net_device *dev = (struct net_device *) GET_OS_DEV(ansDev);
    struct ifreq ifr;
    IANS_BD_PARAM_IVLAN_TABLE request;

    if (vlanIdTable == NULL) {
        IANS_PRINT_ERROR("vlanIdTable is NULL\n");
        return IANS_IOCTL_FAILED;
    }

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return IANS_IOCTL_FAILED;
    }

    if(dev->do_ioctl == NULL) {
        IANS_PRINT_ERROR("%s: dev->do_ioctl == NULL\n", dev->name);
        return IANS_IOCTL_FAILED;
    }

    request.Header.Opcode = IANS_OP_IVLAN_ID_SET_TABLE;

    request.VLanIDNum = vlanNum;
    request.VLanIDTable = vlanIdTable;

    ifr.ifr_data = (void *) &request;

    if((dev->do_ioctl(dev, &ifr, IANS_BASE_SIOC)) != 0) {
        IANS_PRINT_ERROR("%s: failed IANS_OP_IVLAN_ID_SET_TABLE\n", dev->name);
        return IANS_IOCTL_FAILED;
    }
    return IANS_OK;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
IANS_STATUS baseGetTaggingCapability(device_t *ansDev, IANS_BD_PARAM_ITAG_CAP *cap)
{
    struct net_device *dev = (struct net_device *) GET_OS_DEV(ansDev);
    struct ifreq ifr;

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return IANS_IOCTL_FAILED;
    }

    if(dev->do_ioctl == NULL) {
        IANS_PRINT_ERROR("%s: dev->do_ioctl == NULL\n", dev->name);
        return IANS_IOCTL_FAILED;
    }

    cap->Header.Opcode = IANS_OP_ITAG_GET_CAPABILITY;

    ifr.ifr_data = (void *) cap;

    if((dev->do_ioctl(dev, &ifr, IANS_BASE_SIOC)) != 0) {
        IANS_PRINT_ERROR("%s: failed IANS_OP_ITAG_GET_CAPABILITY\n", dev->name);
        return IANS_IOCTL_FAILED;
    }
    return IANS_OK;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
IANS_STATUS baseSetTaggingMode(device_t *ansDev, IANS_BD_TAGGING_MODE TagMode)
{
    struct net_device *dev = (struct net_device *) GET_OS_DEV(ansDev);
    struct ifreq ifr;
    IANS_BD_PARAM_ITAG_SET_MODE request;

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return IANS_IOCTL_FAILED;
    }

    if(dev->do_ioctl == NULL) {
        IANS_PRINT_ERROR("%s: dev->do_ioctl == NULL\n", dev->name);
        return IANS_IOCTL_FAILED;
    }

    request.Header.Opcode = IANS_OP_ITAG_SET_MODE;
    request.SetTagMode = TagMode;

    ifr.ifr_data = (void *) &request;

    if((dev->do_ioctl(dev, &ifr, IANS_BASE_SIOC)) != 0) {
        IANS_PRINT_ERROR("%s: failed IANS_OP_ITAG_SET_MODE\n", dev->name);
        return IANS_IOCTL_FAILED;
    }
    return IANS_OK;
}


/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static void iAnsNotify(struct net_device *dev, int IndType)
{
    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return;
    }
    
    if (dev->name == NULL) {
        IANS_PRINT_ERROR("dev->name == NULL\n");
        return;
    }

    ANSNotify(dev->name, IndType);
}

/*----------------------- MEMBER DEVICE ENTRY POINTS -----------------------*/

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static IANS_STATUS baseOpenDevice(device_t *ansDev)
{
    struct net_device *dev = (struct net_device *) GET_OS_DEV(ansDev);
    BOOLEAN ioctlContext = ANSGetConfigExecuteCommand();
    IANS_STATUS res = IANS_OK;

    if (dev == NULL) {
           IANS_PRINT_ERROR("dev == NULL\n");
           return IANS_MEMBER_OPEN_FAILED;
    }

#ifdef LINUX_KERNEL_24X
    if(!ioctlContext) {
        rtnl_lock();
    }

    if(dev_open(dev) != 0) {
        IANS_PRINT_ERROR("%s: failed to open\n", dev->name);
        res = IANS_MEMBER_OPEN_FAILED;
    }

    if(!ioctlContext) {
        rtnl_unlock();
    }
#endif //LINUX_KERNEL_24X
#ifdef LINUX_KERNEL_22X
    if(dev->open == NULL) {
        return IANS_ERROR;
    }
    if(dev->open(dev) != 0) {
        IANS_PRINT_ERROR("%s: failed to open\n", dev->name);
        return IANS_MEMBER_OPEN_FAILED;
    }
#endif //LINUX_KERNEL_22X

    return res;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static IANS_STATUS baseCloseDevice(device_t *ansDev)
{
    struct net_device *dev = (struct net_device *) GET_OS_DEV(ansDev);
    BOOLEAN ioctlContext = ANSGetConfigExecuteCommand();
    IANS_STATUS res = IANS_OK;

    if (dev == NULL) {
       IANS_PRINT_ERROR("dev == NULL\n");
       return IANS_MEMBER_CLOSE_FAILED;
    }

#ifdef LINUX_KERNEL_24X
    if(!ioctlContext) {
        rtnl_lock();
    }

    if(dev_close(dev) != 0) {
        res = IANS_MEMBER_CLOSE_FAILED;
    }

    if(!ioctlContext) {
        rtnl_unlock();
    }
#endif //LINUX_KERNEL_24X
#ifdef LINUX_KERNEL_22X
    if(dev->stop == NULL) {
        return IANS_ERROR;
    }
    if(dev->stop(dev) != 0) {
        return IANS_MEMBER_CLOSE_FAILED;
    }
#endif //LINUX_KERNEL_22X
    return res;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static IANS_STATUS baseSend(device_t *ansDev, message_t *msg)
{
    struct net_device *dev = (struct net_device *) GET_OS_DEV(ansDev);
    struct sk_buff *skb = (struct sk_buff *) GET_OS_MSG(msg);

    if (dev == NULL) {
       IANS_PRINT_ERROR("dev == NULL\n");
       return IANS_ERROR;
    }

    if (skb == NULL) {
       IANS_PRINT_ERROR("skb == NULL\n");
       return IANS_ERROR;
    }

    skb->dev = dev;

#ifdef LINUX_KERNEL_24X
    dev_queue_xmit(skb);
#endif //LINUX_KERNEL_24X
#ifdef LINUX_KERNEL_22X
    if(dev->hard_start_xmit == NULL) {
        return IANS_ERROR;
    }

    if(dev->hard_start_xmit(skb, dev) != 0) {
        return IANS_ERROR;
    }
#endif //LINUX_KERNEL_22X

    return IANS_OK;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static pstats_t baseGetStats(device_t *ansDev)
{
    struct net_device *dev = (struct net_device *) GET_OS_DEV(ansDev);
    struct net_device_stats *stats;

    if (dev == NULL) {
       IANS_PRINT_ERROR("dev == NULL\n");
       return NULL;
    }

    if(dev->get_stats == NULL) {
        IANS_PRINT_DEBUG("%s: dev->get_stats == NULL\n", dev->name);
        return NULL;
    }

    stats = dev->get_stats(dev);

    if(stats == NULL) {
        IANS_PRINT_DEBUG("%s: failed to get statistics\n", dev->name);
        return NULL;
    }

    return(pstats_t) stats;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static IANS_STATUS baseSetMulticastList(device_t *ansDev, MULTICAST_LIST *multicastList, PACKET_FILTER packetFilter)
{
    struct net_device *dev = (struct net_device *) GET_OS_DEV(ansDev) ;
    int res;

    if (dev == NULL) {
       IANS_PRINT_ERROR("dev == NULL\n");
       return IANS_IOCTL_FAILED;
    }

    if(dev->set_multicast_list == NULL) {
        IANS_PRINT_DEBUG("%s: dev->set_multicast_list == NULL\n", dev->name);
        return IANS_IOCTL_FAILED;
    }

    _freeOSMCList(&dev->mc_list, &dev->mc_count);

    res = _convertAnsMCListToOsMCList(&dev->mc_list, &dev->mc_count, multicastList);
    if(res != IANS_OK) {
        IANS_PRINT_DEBUG("%s: _convertAnsMCListToOsMCList failed!!!\n", dev->name);
        return res;
    }

    ASSERT(dev->mc_count == multicastList->NumAddresses);

    _convertAnsPFToOsPF(&dev->flags, packetFilter);

    dev->set_multicast_list(dev);

    _freeOSMCList(&dev->mc_list, &dev->mc_count);

    return IANS_OK;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static IANS_STATUS baseSetMacAddr(device_t *ansDev, sockAddr_t *addr)
{
    struct net_device *dev = (struct net_device *) GET_OS_DEV(ansDev);

    ASSERT(addr);

    if (dev == NULL) {
       IANS_PRINT_ERROR("dev == NULL\n");
       return IANS_IOCTL_FAILED;
    }

    if(dev->set_mac_address == NULL) {
        IANS_PRINT_DEBUG("%s: dev->set_mac_address == NULL\n", dev->name);
        return IANS_IOCTL_FAILED;
    }

    if(dev->set_mac_address(dev, (struct sockaddr *) addr) != 0) {
        IANS_PRINT_DEBUG("%s: failed to set mac address\n", dev->name);
        return IANS_IOCTL_FAILED;
    }

    return IANS_OK;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static IANS_STATUS baseDoIoctl(device_t *ansDev, preq_t ifr, int cmd)
{
    struct net_device *dev = (struct net_device *) GET_OS_DEV(ansDev);

    if (dev == NULL) {
       IANS_PRINT_ERROR("dev == NULL\n");
       return IANS_IOCTL_FAILED;
    }

    if (ifr == NULL) {
       IANS_PRINT_ERROR("ifr == NULL\n");
       return IANS_IOCTL_FAILED;
    }

    if(dev->do_ioctl == NULL) {
        IANS_PRINT_DEBUG("%s: dev->do_ioctl == NULL\n", dev->name);
        return IANS_IOCTL_FAILED;
    }

    if(dev->do_ioctl(dev, (struct ifreq *)ifr, cmd) != 0) {
        IANS_PRINT_DEBUG("%s: did not recognize generic ioctl 0x%04X\n", dev->name, cmd);
        return IANS_IOCTL_FAILED;
    }

    return IANS_OK;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static IANS_STATUS baseChangeMtu(device_t *ansDev, int new_mtu)
{
    struct net_device *dev = (struct net_device *) GET_OS_DEV(ansDev) ;

    if (dev == NULL) {
       IANS_PRINT_ERROR("dev == NULL\n");
       return IANS_IOCTL_FAILED;
    }

    if(new_mtu == dev->mtu) {
        //IANS_PRINT_DEBUG("%s: new_mtu == dev->mtu, nothing to do\n", dev->name);
        return IANS_OK;
    }

    if(new_mtu < 0) {
        IANS_PRINT_DEBUG("%s: new_mtu must be positive\n", dev->name);
        return IANS_IOCTL_FAILED;
    }

    if(dev->change_mtu) {
        if(dev->change_mtu(dev, new_mtu) != 0) {
            IANS_PRINT_DEBUG("%s: failed to change mtu\n", dev->name);
            return IANS_IOCTL_FAILED;
        }
    }
    else {
        if((new_mtu < ETHERNET_MIN_FRAME_SIZE-ETHERNET_HEADER_SIZE) ||
           (new_mtu > ETHERNET_MAX_FRAME_SIZE-ETHERNET_HEADER_SIZE)
          ) {
            IANS_PRINT_DEBUG("%s: mtu out of bounds\n", dev->name);
            return IANS_IOCTL_FAILED;
        }
        else {
            dev->mtu = new_mtu;
        }
    }

    return IANS_OK;
}

/*-------------------------- SERVICE FUNCTIONS -----------------------------*/

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static void _freeOSMCList(struct dev_mc_list **mcList, int *mcCount)
{
    struct dev_mc_list* tempMCList;

    while(*mcList) {
        tempMCList = *mcList;
        *mcList = (*mcList)->next;
        iNCGGPMemFree((void **)(&tempMCList));
    }

    *mcList = NULL;
    *mcCount = 0;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static IANS_STATUS _addOSMCList(struct dev_mc_list **mcList, int *mcCount)
{
    struct dev_mc_list* tempMCList = NULL;

    iNCGGPSizedMemAlloc((void **)(&tempMCList), sizeof(struct dev_mc_list));
    if(tempMCList == NULL) {
        return IANS_MEM_ERROR;
    }

    tempMCList->next = *mcList;
    *mcList = tempMCList;
    *mcCount = *mcCount + 1;
    return IANS_OK;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static inline IANS_STATUS _convertAnsMCListToOsMCList(struct dev_mc_list **osList,
                                             int *osCount,
                                             MULTICAST_LIST *ansList)
{
    MULTICAST_ENTRY *currAnsEntry = ansList->pAddressesList;
    struct dev_mc_list *currOsList = NULL;
    int addrCount = 0;
    int res;

    while(currAnsEntry) {
        res = _addOSMCList(&currOsList, &addrCount);
        if(res != IANS_OK) {
            IANS_PRINT_DEBUG("_addOSMCList failed!!!\n");
            _freeOSMCList(&currOsList, &addrCount);
            return res;
        }

        iNCGMacAddrCopy(currOsList->dmi_addr, currAnsEntry->MacAddr.MacAddr);
        currOsList->dmi_addrlen = currAnsEntry->dmiAddrLen;
        currOsList->dmi_users = currAnsEntry->dmiUsers;
        currOsList->dmi_gusers = currAnsEntry->dmiGusers;

        currAnsEntry = currAnsEntry->pNextEntry;
    }

    *osCount = addrCount;
    *osList = currOsList;
    return IANS_OK;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static inline void _convertAnsPFToOsPF(unsigned short *osFlags,
                                     PACKET_FILTER packetFilter)
{
    if(packetFilter & PF_PROMISCUOUS_BIT) {
        *osFlags |= IFF_PROMISC;
    }
    else {
        *osFlags &= ~(IFF_PROMISC);
    }

    if(packetFilter & PF_ALLMCA_BIT) {
        *osFlags |= IFF_ALLMULTI;
    }
    else {
        *osFlags &= ~(IFF_ALLMULTI);
    }
}

#ifdef LINUX_KERNEL_24X
/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static void _convertEthToolToStats(struct ethtool_cmd *ecmd, IANS_BD_IOC_PARAM_STATUS *stat)
{
    switch(ecmd->duplex) {
        case DUPLEX_FULL:
            stat->Status.Duplex = IANS_STATUS_DUPLEX_FULL;
            break;
        case DUPLEX_HALF:
            stat->Status.Duplex = IANS_STATUS_DUPLEX_HALF;
            break;
        default:
            stat->Status.Duplex = IANS_STATUS_DUPLEX_NOT_SUPPORTED;
            break;
    }

    switch(ecmd->speed) {
        case SPEED_10:
            stat->Status.LinkSpeed = IANS_STATUS_LINK_SPEED_10MBPS;
            break;
        case SPEED_100:
            stat->Status.LinkSpeed = IANS_STATUS_LINK_SPEED_100MBPS;
            break;
        case SPEED_1000:
            stat->Status.LinkSpeed = IANS_STATUS_LINK_SPEED_1000MBPS;
            break;
        default:
            stat->Status.LinkSpeed = IANS_STATUS_LINK_SPEED_NOT_SUPPORTED;
            break;
    }

    if(stat->Status.LinkSpeed != IANS_STATUS_LINK_SPEED_NOT_SUPPORTED) {
        stat->Status.LinkStatus = IANS_STATUS_LINK_OK;
    }
    else {
        stat->Status.LinkStatus = IANS_STATUS_LINK_FAIL;
    }

    stat->Status.StatusVersion = IANS_STATUS_VERSION;
    stat->Status.HardwareFailure = IANS_STATUS_HARDWARE_NOT_SUPPORTED;
    stat->Status.DuringResetProcess = IANS_STATUS_RESET_NOT_SUPPORTED;
    stat->Status.Suspended = IANS_STATUS_SUSPENDED_NOT_SUPPORTED;
    stat->Status.HotPlug = IANS_STATUS_HOT_PLUG_NOT_SUPPORTED;
}
#endif //LINUX_KERNEL_24X

