/*******************************************************************************
* This software program is available to you under a choice of one of two 
* licenses. You may choose to be licensed under either the GNU General Public 
* License 2.0, June 1991, available at http://www.fsf.org/copyleft/gpl.html, 
* or the Intel BSD + Patent License, the text of which follows:
* 
* Recipient has requested a license and Intel Corporation ("Intel") is willing
* to grant a license for the software entitled Linux Intel Advanced Network
* Services (iANS) (the "Licensed Program") being provided by Intel Corporation.
* The following definitions apply to this license:
* 
* "Licensed Patents" means patent claims licensable by Intel Corporation which 
* are necessarily infringed by the use of sale of the Software alone or when 
* combined with the operating system referred to below.
* 
* "Recipient" means the party to whom Intel delivers this Software.
* 
* "Licensee" means Recipient and those third parties that receive a license to 
* any operating system available under the GNU General Public License 2.0 or 
* later.
* 
* Copyright (c) 1999 - 2002 Intel Corporation.
* All rights reserved.
* 
* The license is provided to Recipient and Recipient's Licensees under the 
* following terms.
* 
* Redistribution and use in source and binary forms of the Software, with or 
* without modification, are permitted provided that the following conditions 
* are met:
* 
* Redistributions of source code of the Software may retain the above 
* copyright notice, this list of conditions and the following disclaimer.
* 
* Redistributions in binary form of the Software may reproduce the above 
* copyright notice, this list of conditions and the following disclaimer in 
* the documentation and/or materials provided with the distribution.
* 
* Neither the name of Intel Corporation nor the names of its contributors 
* shall be used to endorse or promote products derived from this Software 
* without specific prior written permission.
* 
* Intel hereby grants Recipient and Licensees a non-exclusive, worldwide, 
* royalty-free patent license under Licensed Patents to make, use, sell, offer 
* to sell, import and otherwise transfer the Software, if any, in source code 
* and object code form. This license shall include changes to the Software 
* that are error corrections or other minor changes to the Software that do 
* not add functionality or features when the Software is incorporated in any 
* version of an operating system that has been distributed under the GNU 
* General Public License 2.0 or later. This patent license shall apply to the 
* combination of the Software and any operating system licensed under the GNU 
* General Public License 2.0 or later if, at the time Intel provides the 
* Software to Recipient, such addition of the Software to the then publicly 
* available versions of such operating systems available under the GNU General 
* Public License 2.0 or later (whether in gold, beta or alpha form) causes 
* such combination to be covered by the Licensed Patents. The patent license 
* shall not apply to any other combinations which include the Software. NO 
* hardware per se is licensed hereunder.
* 
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
* AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
* IMPLIED WARRANTIES OF MECHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
* ARE DISCLAIMED. IN NO EVENT SHALL INTEL OR IT CONTRIBUTORS BE LIABLE FOR ANY 
* DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
* (INCLUDING, BUT NOT LIMITED, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
* ANY LOSS OF USE; DATA, OR PROFITS; OR BUSINESS INTERUPTION) HOWEVER CAUSED 
* AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY OR 
* TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/

/*****************************************************************************
******************************************************************************
**                                                                          **
** INTEL CORPORATION                                                        **
**                                                                          **
** This software is supplied under the terms of the license included        **
** above.  All use of this software must be in accordance with the terms    **
** of that license.                                                         **
**                                                                          **
** THIS FILE IS USED WHEN CREATING ians_core.o. HENCE, IT SHOULD NOT BE     **
** MODIFIED!                                                                **
**                                                                          **
**  Module Name:    ians_kernel.c                                           **
**                                                                          **
**  Abstract:       This module contains all the functions needed for the   **
**                  interface of the iANS module with the kernel.           **
**                                                                          **
**  Environment:    Kernel Mode (linux 2.2.x, 2.4.x)                        **
**                                                                          **
**                  Written using Tab Size = 4, Indent Size = 4             **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
******************************************************************************
*****************************************************************************/

#define _IANS_MAIN_MODULE_C_

#include <linux/init.h>
#include <linux/module.h>
#include <linux/netdevice.h>
#include <linux/skbuff.h>
#include <linux/inetdevice.h>
#include "ians_kcompat.h"
#include "ians_status.h"
#include "ans_interface.h"
#include "lib/incg_open_utils.h"
#include "lib/incg_gp_mem.h"
#include "lib/incg_defs.h"
#include "lib/incg_net.h"
#include "lib/incg_types.h"
#include "lib/incg_flow.h"
#include "lib/incg_locks.h"
#include "base_comm.h"

#ifdef LINUX_KERNEL_24X
#include <linux/notifier.h>
#endif //LINUX_KERNEL_24X


MODULE_AUTHOR("Intel Corporation, <linux.nics@intel.com>");
MODULE_DESCRIPTION("Intel(R) Advanced Network Services");
MODULE_LICENSE("Proprietary");


/*****************************************************************************
**                  F o r w a r d  D e c l e r a t i o n s                  **
*****************************************************************************/

/*---------------------- VADAPTER DEVICE ENTRY POINTS ----------------------*/
static int iansVadapterOpen(struct net_device *dev);
static int iansVadapterStop(struct net_device *dev);
static int iansVadapterSend(struct sk_buff *skb, struct net_device *dev);
static struct net_device_stats *iansVadapterGetStats(struct net_device *dev);
static void iansVadapterSetMulticastList(struct net_device *dev);
static int iansVadapterSetMacAddress(struct net_device *dev, void *addr);
static int iansVadapterDoIoctl(struct net_device *dev, struct ifreq *rq, int cmd);
static int iansVadapterChangeMtu(struct net_device *dev, int new_mtu);

/*----------------------- CONTROL DEVICE ENTRY POINTS ----------------------*/
static int iansControlDoIoctl(struct net_device *dev, struct ifreq *rq, int cmd);
static int iansControlOpen(struct net_device *dev);
static int iansControlStop(struct net_device *dev);
static int iansControlSend(struct sk_buff *skb, struct net_device *dev);
static struct net_device_stats *iansControlGetStats(struct net_device *dev);
static void iansControlSetMCList(struct net_device *dev);
static int iansControlSetMacAddress(struct net_device *dev, void *addr);
static int iansControlChangeMtu(struct net_device *dev, int new_mtu);

/*------------------------ REGISTERED FUNCTIONS ----------------------------*/
static int iansOpenReceive(struct sk_buff *skb, struct net_device *dev, struct packet_type *pt);
#ifdef LINUX_KERNEL_24X
static int ians_netdev_event(struct notifier_block *nb, unsigned long notification, void *ptr);
#endif //LINUX_KERNEL_24X
static int ians_inetaddr_event(struct notifier_block *nb, unsigned long notification, void *ptr);

/*-------------------------- SERVICE FUNCTIONS -----------------------------*/
static struct net_device* _iansDevAlloc(const char *name);
static struct net_device* _iansInitEtherdev(const char *name);
static inline IANS_STATUS _convertOsMCListToAnsMCList(struct dev_mc_list *osList, int osCount, MULTICAST_LIST *ansList);
static inline void _convertOsPFToAnsPF(USHORT osFlags, PACKET_FILTER *packetFilter);


/*****************************************************************************
******************************************************************************
**                  G L O B A L  P A R A M E T E R S                        **
******************************************************************************
*****************************************************************************/
static struct net_device *control_g = NULL;

#ifdef LINUX_KERNEL_22X
static struct packet_type iansPacketType = {
    IANS_FRAME_TYPE, // no need for htons !!!
    NULL,            // Null = All devices
    iansOpenReceive, // receive function
    NULL,            // Private to the packet type
    NULL,            // next
};
#endif //LINUX_KERNEL_22X

#ifdef LINUX_KERNEL_24X
static struct packet_type iansPacketType = {
    type:   IANS_FRAME_TYPE,// no need for htons !!!
    func:   iansOpenReceive,// receive function
    data:   (void*)1,       // understand shared skbs
};
#endif //LINUX_KERNEL_24X

#ifdef LINUX_KERNEL_24X
static struct notifier_block ians_netdev_notifier = {
    notifier_call: ians_netdev_event,
};
#endif //LINUX_KERNEL_24X

static struct notifier_block ians_inetaddr_notifier = {
    notifier_call: ians_inetaddr_event,
}; 

/*---------------------- VADAPTER DEVICE FUNCTIONS -------------------------*/

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
IANS_STATUS iansInitVadapterDev(const char *name, device_t **ansVadapterDev)
{
    struct net_device *osVadapterDev;

    ASSERT(name);

    if((osVadapterDev = _iansInitEtherdev(name)) == NULL) {
        return IANS_VADAPTER_INITIALIZE_FAIL;
    }

    osVadapterDev->open = iansVadapterOpen;
    osVadapterDev->stop = iansVadapterStop;
    osVadapterDev->hard_start_xmit = iansVadapterSend;
    osVadapterDev->get_stats = iansVadapterGetStats;
    osVadapterDev->set_multicast_list = iansVadapterSetMulticastList;
    osVadapterDev->set_mac_address = iansVadapterSetMacAddress;
    osVadapterDev->do_ioctl = iansVadapterDoIoctl;
    osVadapterDev->change_mtu = iansVadapterChangeMtu;
    osVadapterDev->tx_queue_len = 100;

#ifdef LINUX_KERNEL_22X
    osVadapterDev->start = 0;
#endif //LINUX_KERNEL_22X

    iNCGGPSizedMemAlloc((void **)(ansVadapterDev), sizeof(device_t));
    if(*ansVadapterDev == NULL) {
        IANS_PRINT_ERROR("Failed to allocate an IANS device structure\n");
        iNCGGPMemFree((void **)(ansVadapterDev));
        return IANS_MEM_ERROR;
    }

    (*ansVadapterDev)->pOsDev = osVadapterDev;
    (*ansVadapterDev)->name = osVadapterDev->name;
    (*ansVadapterDev)->mtu = &(osVadapterDev->mtu);
    (*ansVadapterDev)->mac = (MAC_ADDR *)(osVadapterDev->dev_addr);

    return IANS_OK;
}


/*---------------------- VADAPTER DEVICE ENTRY POINTS ----------------------*/

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int iansVadapterOpen(struct net_device *dev)
{
    device_t *ansDev = NULL;
    int res = 0;
    IANS_STATUS ansRes = IANS_OK;

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return -EAGAIN;
    }

    if (dev->name == NULL) {
        IANS_PRINT_ERROR("dev->name == NULL\n");
        return -EAGAIN;
    }

    ANSConfigLockBH();

    ansDev = ANSGetVadapterAnsDevice(dev->name);
    if(ansDev == NULL) {
        IANS_PRINT_ERROR("there is no such ans device.\n");
        ANSConfigUnlockBH();
        return -EAGAIN;
    }

    ansRes = ansDev->open(ansDev);
    if(ansRes == IANS_OK) {
        MOD_INC_USE_COUNT;

#ifdef LINUX_KERNEL_22X
        dev->start = 1;
#endif //LINUX_KERNEL_22X
    
    }
    else {
        res = -EPERM; //operation not permitted
    }

    ANSConfigUnlockBH();

    return res;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int iansVadapterStop(struct net_device *dev)
{
    device_t *ansDev = NULL;
    IANS_STATUS ansRes = IANS_OK;

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return -EAGAIN;
    }

    if (dev->name == NULL) {
        IANS_PRINT_ERROR("dev->name == NULL\n");
        return -EAGAIN;
    }

    ANSConfigLockBH();

    ansDev = ANSGetVadapterAnsDevice(dev->name);
    if(ansDev == NULL) {
        IANS_PRINT_ERROR("there is no such ans device.\n");
        ANSConfigUnlockBH();
        return -EAGAIN;
    }

    ansRes = ansDev->close(ansDev);
    if(ansRes == IANS_OK) {
        MOD_DEC_USE_COUNT;

#ifdef LINUX_KERNEL_22X
        dev->start = 0;
#endif //LINUX_KERNEL_22X
    
    }

    ANSConfigUnlockBH();

    return 0;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int iansVadapterSend(struct sk_buff *skb, struct net_device *dev)
{
    device_t *ansDev;
    message_t msg; //automatic variable to save allocation time.
    int res = 0;
    int ansRes = IANS_OK;

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return -EAGAIN;
    }

    if (skb == NULL) {
            IANS_PRINT_ERROR("skb == NULL\n");
            return -EAGAIN;
    }

    if (dev->name == NULL) {
        IANS_PRINT_ERROR("dev->name == NULL\n");
        return -EAGAIN;
    }

    ANSConfigLock();

    ansDev = ANSGetVadapterAnsDevice(dev->name);
    if(ansDev == NULL) {
        IANS_PRINT_ERROR("there is no such ans device.\n");
        ANSConfigUnlock();
        return -EAGAIN;
    }

    iAnsMemSet(&msg, 0, sizeof(message_t));

    iansConvertMessageToAnsMessage(skb, &msg, TRANSMIT_FLOW);

    ansRes = ansDev->send(&msg, ansDev);
    if(ansRes != IANS_OK) {
        res = 1; //The base driver return value in case of failure
    }

    ANSConfigUnlock();

    return res;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static struct net_device_stats *iansVadapterGetStats(struct net_device *dev)
{
    device_t *ansDev = NULL;
    struct net_device_stats *res = NULL;

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return NULL;
    }

    if (dev->name == NULL) {
        IANS_PRINT_ERROR("dev->name == NULL\n");
        return NULL;
    }

    ANSConfigLockBH();

    ansDev = ANSGetVadapterAnsDevice(dev->name);
    if(ansDev == NULL) {
        IANS_PRINT_ERROR("there is no such ans device.\n");
        ANSConfigUnlockBH();
        return NULL;
    }

    res = (struct net_device_stats *) ansDev->getStats(ansDev);

    ANSConfigUnlockBH();

    return res;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static void iansVadapterSetMulticastList(struct net_device *dev)
{
    MULTICAST_LIST multicastList;
    PACKET_FILTER packetFilter;
    device_t *ansDev = NULL;
    IANS_STATUS res;

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return;
    }

    if (dev->name == NULL) {
        IANS_PRINT_ERROR("dev->name == NULL\n");
        return;
    }

    memset(&multicastList, 0, sizeof(MULTICAST_LIST));

    if(dev->mc_list) {
        ASSERT(dev->mc_count);
        res = _convertOsMCListToAnsMCList(dev->mc_list, dev->mc_count, &multicastList);
        if(res != IANS_OK) {
            return;
        }
    }

    _convertOsPFToAnsPF(dev->flags, &packetFilter);

    ANSConfigLockBH();

    ansDev = ANSGetVadapterAnsDevice(dev->name);
    if(ansDev == NULL) {
        IANS_PRINT_ERROR("there is no such ans device.\n");
        ANSConfigUnlockBH();
        return;
    }

    res = ansDev->setMulticastList(ansDev, &multicastList, packetFilter);
    if(res != IANS_OK) {
        IANS_PRINT_ERROR("Set multicast list failed!\n");
        ANSConfigUnlockBH();
        return;
    }

    ANSConfigUnlockBH();
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int iansVadapterSetMacAddress(struct net_device *dev, void *addr)
{
    device_t *ansDev = NULL;
    sockAddr_t *sockaddr;
    IANS_STATUS res;

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return -EAGAIN;
    }

    if (dev->name == NULL) {
        IANS_PRINT_ERROR("dev->name == NULL\n");
        return -EAGAIN;
    }

    if (addr == NULL) {
        IANS_PRINT_ERROR("addr == NULL\n");
        return -EAGAIN;
    }

    ANSConfigLockBH();

    ansDev = ANSGetVadapterAnsDevice(dev->name);
    if(ansDev == NULL) {
        IANS_PRINT_ERROR("there is no such ans device.\n");
        ANSConfigUnlockBH();
        return -EAGAIN;
    }

    sockaddr = (sockAddr_t*)addr;

    res = ansDev->setMacAddr(ansDev, sockaddr);
    if(res != IANS_OK) {
        IANS_PRINT_ERROR("Set MAC address failed!\n");
        ANSConfigUnlockBH();
        return -1;
    }

    ANSConfigUnlockBH();
    return 0;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int iansVadapterDoIoctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
    device_t *ansDev = NULL;
    IANS_STATUS res;

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return -EAGAIN;
    }

    if (dev->name == NULL) {
        IANS_PRINT_ERROR("dev->name == NULL\n");
        return -EAGAIN;
    }

    if (rq == NULL) {
        IANS_PRINT_ERROR("rq == NULL\n");
        return -EAGAIN;
    }

    if((cmd >= SIOCDEVPRIVATE) && (cmd <(SIOCDEVPRIVATE+16))) {
        // Private IOCTL's handling is undefined. reject them.
        IANS_PRINT_DEBUG("vadapter %s got a private IOCTL 0x%04X\n", dev->name, cmd);
        return 0;
    }

    ANSConfigLockBH();

    ansDev = ANSGetVadapterAnsDevice(dev->name);
    if(ansDev == NULL) {
        IANS_PRINT_ERROR("there is no such ans device.\n");
        ANSConfigUnlockBH();
        return -EAGAIN;
    }

    res = ansDev->doIoctl(ansDev, rq, cmd);

    ANSConfigUnlockBH();

    return res;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int iansVadapterChangeMtu(struct net_device *dev, int new_mtu)
{
    device_t *ansDev = NULL;
    int res;

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return -EAGAIN;
    }

    if (dev->name == NULL) {
        IANS_PRINT_ERROR("dev->name == NULL\n");
        return -EAGAIN;
    }

    if(new_mtu == dev->mtu) {
        //nothing to do
        return 0;
    }

    if(new_mtu < 0) {
        //must be positive
        return -EINVAL;
    }

    ANSConfigLockBH();

    ansDev = ANSGetVadapterAnsDevice(dev->name);
    if(ansDev == NULL) {
        IANS_PRINT_ERROR("there is no such ans device.\n");
        ANSConfigUnlockBH();
        return -EAGAIN;
    }

    res = ansDev->changeMtu(ansDev, new_mtu);
    if(res != IANS_OK) {
        IANS_PRINT_ERROR("Change MTU failed!\n");
        ANSConfigUnlockBH();
        return -EINVAL;
    }

    ANSConfigUnlockBH();
    return 0;
}


/*---------------------- CONTROL DEVICE FUNCTIONS --------------------------*/

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int iansControlDoIoctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
    IANS_STATUS res = IANS_OK;
    char *data = rq->ifr_ifru.ifru_data;

    if(dev != control_g) {
        IANS_PRINT_ERROR("Got control IOCTL on non control device: dev->name %s\n", dev->name);
        return IANS_OS_ERROR;
    }

    if (rq == NULL) {
        IANS_PRINT_ERROR("rq == NULL\n");
        return IANS_OS_ERROR;
    }

    switch(cmd) {
        case IANSIOCCONFIG:
            if(data == NULL) {
                IANS_PRINT_ERROR("Got null data in control IOCTL");
                res = IANS_OS_ERROR;
                break;
            }

            ANSSetConfigExecuteCommand(1);
            res = ANSDoConfigIoctl(data);
            ANSSetConfigExecuteCommand(0);
            break;

        case IANSIOCSERVICE:
            ANSSetConfigExecuteCommand(1);
            res = ANSDoServiceIoctl(data);
            ANSSetConfigExecuteCommand(0);
            break;

        default:
            IANS_PRINT_DEBUG("Ioctl not supported");
            res = IANS_OS_ERROR;
            break;
    }

    return res;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int iansControlOpen(struct net_device *dev)
{
    IANS_PRINT_DEBUG("Got open on control device\n");
    return -EPERM;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int iansControlStop(struct net_device *dev)
{
    IANS_PRINT_DEBUG("Got stop on control device\n");
    return -EPERM;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int iansControlSend(struct sk_buff *skb, struct net_device *dev)
{
    IANS_PRINT_DEBUG("Got hard_start_xmit for control device\n");
    return 0;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static struct net_device_stats *iansControlGetStats(struct net_device *dev)
{
    //IANS_PRINT_DEBUG("Got get_statistics for control device\n");
    return NULL;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static void iansControlSetMCList(struct net_device *dev)
{
    IANS_PRINT_DEBUG("Got set_multicast_list for control device\n");
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int iansControlSetMacAddress(struct net_device *dev, void *addr)
{
    IANS_PRINT_DEBUG("Got set_mac_address for control device\n");
    return -EPERM;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int iansControlChangeMtu(struct net_device *dev, int new_mtu)
{
    IANS_PRINT_DEBUG("Got change_mtu for control device\n");
    return -EPERM;
}


/*------------------------ REGISTERED FUNCTIONS ----------------------------*/

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int iansOpenReceive(struct sk_buff *skb, struct net_device *dev, struct packet_type *pt)
{
    IANS_FLOW_STATUS res;
    message_t msg; //automatic variable to save allocation time.

    if (skb == NULL) {
       IANS_PRINT_ERROR("skb == NULL\n");
       return NET_RX_BAD;
    }
    
    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");       
        kfree_skb(skb);
        return NET_RX_BAD;
    }

    if (dev->name == NULL) {
        IANS_PRINT_ERROR("dev->name == NULL\n");
        kfree_skb(skb);
        return NET_RX_BAD;
    }

#ifdef LINUX_KERNEL_24X
    if ((skb = skb_share_check(skb, GFP_ATOMIC)) == NULL) {
        return NET_RX_DROP;
    }
#endif

    if(skb->protocol != IANS_FRAME_TYPE) {
        IANS_PRINT_ERROR("Protocol of received sk_buff differs from the IANS frame type!\n");
        IANS_PRINT_ERROR("Protocol is set to %x\n", skb->protocol);
        kfree_skb(skb);
        return NET_RX_BAD;
    }

    iAnsMemSet(&msg, 0, sizeof(message_t));
    iansConvertMessageToAnsMessage(skb, &msg, RECEIVE_FLOW);

    ANSConfigLock();

    res = ANSReceive(&msg, dev->name, (pPacket_t) pt);

    ANSConfigUnlock();

    if(res == FLOW_CONTINUE) {
        netif_rx(skb);
        return NET_RX_SUCCESS;
    }

    // anything else is considered an error
    switch(res) {
        case FLOW_DOESNT_EXIST:
            IANS_PRINT_DEBUG("flow doesn't exist\n");
            break;
        case FLOW_DROP:
            IANS_PRINT_DEBUG("packet dropped\n");
            break;
        case FLOW_INTERNAL_ERROR:
            IANS_PRINT_DEBUG("internal error\n");
            break;
        case FLOW_WRONG_STATE:
            IANS_PRINT_DEBUG("flow in wrong state\n");
            break;
        default:
            break;
    }

    kfree_skb(skb);
    return NET_RX_DROP;
}

#ifdef LINUX_KERNEL_24X
/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int ians_netdev_event(struct notifier_block *nb, unsigned long notification, void *ptr)
{
    IANS_STATUS res = IANS_OK;
    struct net_device *dev = NULL;

    if (ptr == NULL) {
       IANS_PRINT_ERROR("bad notification parameters\n");
       return NOTIFY_BAD;
    }

#ifdef IANS_DEBUG
    /*
    static char* NotifyStrings[] = {
         "",
         "NETDEV_UP",
         "NETDEV_DOWN",
         "NETDEV_REBOOT",
         "NETDEV_CHANGE",
         "NETDEV_REGISTER",
         "NETDEV_UNREGISTER",
         "NETDEV_CHANGEMTU",
         "NETDEV_CHANGEADDR",
         "NETDEV_GOING_DOWN",
         "NETDEV_CHANGENAME"
    };
    */
#endif //IANS_DEBUG

    dev = (struct net_device*)ptr;
    if(dev && (notification>=NETDEV_UP) && (notification<=NETDEV_CHANGENAME)) {
        //IANS_PRINT_DEBUG("Got %s notification from %s\n", NotifyStrings[notification], dev->name);
        res = ANSHandleNetDevEvent(dev->name, notification);
        if(res != IANS_OK) {
            IANS_PRINT_ERROR("couldn't handle notification");
        }
    }
    return NOTIFY_DONE;
}
#endif //LINUX_KERNEL_24X


/*****************************************************************************
**                                                                          **
**  Function Name: ians_inetaddr_event                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int ians_inetaddr_event(struct notifier_block *nb, unsigned long notification, void *ptr)
{
    IANS_STATUS res = IANS_OK;
    struct in_ifaddr *ifa = (struct in_ifaddr *)ptr;
    struct net_device *dev = NULL;
    
    if (ifa){
        dev = ifa->ifa_dev->dev;
    }
    else {
        IANS_PRINT_ERROR("bad notification parameters\n");
        return NOTIFY_BAD;
    }

#ifdef IANS_DEBUG
    /*
    static char* NotifyStrings[] = {
         "",
         "NETDEV_UP",
         "NETDEV_DOWN",
         "NETDEV_REBOOT",
         "NETDEV_CHANGE",
         "NETDEV_REGISTER",
         "NETDEV_UNREGISTER",
         "NETDEV_CHANGEMTU",
         "NETDEV_CHANGEADDR",
         "NETDEV_GOING_DOWN",
         "NETDEV_CHANGENAME"
    };
    */
#endif //IANS_DEBUG

    if(dev && ((notification==NETDEV_UP) || (notification==NETDEV_DOWN))) {
        //IANS_PRINT_DEBUG("Got %s notification from %s\n", NotifyStrings[notification], dev->name);
        res = ANSHandleInetAddrEvent(dev->name, notification,ifa->ifa_address);
        if(res != IANS_OK) {
            IANS_PRINT_ERROR("couldn't handle notification");
        }
    }
    return NOTIFY_DONE;
}


/*--------------------------- MODULE FUNCTIONS -----------------------------*/

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
int ians_init_module(void)
{
    IANS_STATUS res = IANS_OK;

    res = ANSConfigCreate();
    if(res!=IANS_OK) {
        return -1;
    }

    if((control_g = _iansInitEtherdev("ians")) == NULL) {
        IANS_PRINT_ERROR("Init etherdev failed - cannot load the IANS module.\n");
        return -1;
    }

    control_g->open = iansControlOpen;
    control_g->stop = iansControlStop;
    control_g->hard_start_xmit = iansControlSend;
    control_g->get_stats = iansControlGetStats;
    control_g->set_multicast_list = iansControlSetMCList;
    control_g->set_mac_address = iansControlSetMacAddress;
    control_g->do_ioctl = iansControlDoIoctl;
    control_g->change_mtu = iansControlChangeMtu;
    control_g->flags = 0;
    control_g->mtu = 0;
    control_g->tx_queue_len = 0;

#ifdef LINUX_KERNEL_22X
    control_g->start = 0;
#endif //LINUX_KERNEL_22X

    SET_MODULE_OWNER(control_g);

    register_netdev(control_g);

#ifdef LINUX_KERNEL_24X
    register_netdevice_notifier(&ians_netdev_notifier);
#endif //LINUX_KERNEL_24X
    register_inetaddr_notifier(&ians_inetaddr_notifier);
   
    dev_add_pack(&iansPacketType);

    return 0;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
void ians_cleanup_module(void)
{
    
    dev_remove_pack(&iansPacketType);
    
    unregister_inetaddr_notifier(&ians_inetaddr_notifier);
#ifdef LINUX_KERNEL_24X
    unregister_netdevice_notifier(&ians_netdev_notifier);
#endif //LINUX_KERNEL_24X

    unregister_netdev(control_g);

    iNCGGPMemFree((void**)(&control_g));

    ANSConfigFree();

#ifdef MEM_DEBUG
    do {
        int i = iNCGGPMemGetAllocCount();
        if(i != 0) {
            IANS_PRINT_DEBUG("Allocation count = %d\n", i);
        }
    } while(0);
#endif //MEM_DEBUG
}


/*-------------------------- SERVICE FUNCTIONS -----------------------------*/

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static struct net_device* _iansDevAlloc(const char *name)
{
    int allocSize;
    struct net_device *dev;
#ifdef LINUX_KERNEL_24X
    struct net_device *temp;
#endif //LINUX_KERNEL_24X

#ifdef LINUX_KERNEL_22X
    if(dev_get_by_name(name) != NULL) {
#endif //LINUX_KERNEL_22X
#ifdef LINUX_KERNEL_24X
    if((temp = dev_get_by_name(name)) != NULL) {
        dev_put(temp);
#endif //LINUX_KERNEL_24X
        IANS_PRINT_ERROR("A device with this name(%s) already exists\n", name);
        return NULL;
    }

#ifdef LINUX_KERNEL_22X
    allocSize = sizeof(struct net_device)+IFNAMSIZ;
#endif //LINUX_KERNEL_22X
#ifdef LINUX_KERNEL_24X
    allocSize = sizeof(struct net_device);
#endif //LINUX_KERNEL_24X
    allocSize += 3;
    allocSize &= ~3;

    iNCGGPSizedMemAlloc((void **)(&dev), allocSize);
    if(dev == NULL) {
        IANS_PRINT_ERROR("kmalloc failed.\n");
        return NULL;
    }

    iAnsMemSet(dev, 0, allocSize);

#ifdef LINUX_KERNEL_22X
    dev->name= (char *)(dev+1);
#endif //LINUX_KERNEL_22X

    iAnsStrCpy(dev->name, name);

    return dev;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static struct net_device* _iansInitEtherdev(const char *name)
{
    struct net_device *dev = NULL;

    if((dev = _iansDevAlloc(name)) == NULL) {
        return NULL;
    }

    ether_setup(dev);

    return dev;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static inline IANS_STATUS _convertOsMCListToAnsMCList(struct dev_mc_list *osList, int osCount, MULTICAST_LIST *ansList)
{
    struct dev_mc_list *currList;
    IANS_FLOW_STATUS res;

    ASSERT(osList && ansList);

    for(currList = osList; currList != NULL; currList = currList->next) {
        res = ANSAddMulticastAddress(ansList,
                                     (MAC_ADDR*)currList->dmi_addr,
                                     currList->dmi_addrlen,
                                     currList->dmi_users,
                                     currList->dmi_gusers);

        if(res != FLOW_OK) {
            if(res == FLOW_GP_RESOURCES) {
                IANS_PRINT_ERROR("kmalloc failed.\n");
            }
            IANS_PRINT_ERROR("Failed in adding a multicast address.\n");
            return IANS_ERROR;
        }
    }

    ASSERT(osCount == ansList->NumAddresses);

    return IANS_OK;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static inline void _convertOsPFToAnsPF(USHORT osFlags, PACKET_FILTER *packetFilter)
{
    if(osFlags & IFF_PROMISC) {
        *packetFilter |= PF_PROMISCUOUS_BIT;
    }
    else {
        *packetFilter &= ~(PF_PROMISCUOUS_BIT);
    }

    if(osFlags & IFF_ALLMULTI) {
        *packetFilter |= PF_ALLMCA_BIT;
    }
    else {
        *packetFilter &= ~(PF_ALLMCA_BIT);
    }
}

/*-------------------------- MODULE ENTRY POINTS ---------------------------*/
module_init(ians_init_module);
module_exit(ians_cleanup_module);

