﻿<#
/// <copyright>
/// INTEL CONFIDENTIAL
///
/// Copyright 2014 Intel Corporation
///
/// This software and the related documents are Intel copyrighted materials, and your use of
/// them is governed by the express license under which they were provided to you ("License").
/// Unless the License provides otherwise, you may not use, modify, copy, publish, distribute,
/// disclose or transmit this software or the related documents without Intel's prior written
/// permission.
///
/// This software and the related documents are provided as is, with no express or implied
/// warranties, other than those that are expressly stated in the License.
///
/// </copyright>
#>

#User input arguments
param
(
    [Parameter(Mandatory=$true)]
    [string]$Action,

    [string]$ConfigPath,

    [switch]$BDF
)

#Global file variables
$Script:context
$Script:service
$Script:lockobj
$script:options
$Script:Adapters = $null
$Script:bSave = $false
$Script:bRestore = $false
$Script:bRemove = $false
$Script:bUpSave = $false
$Script:bUpRestore = $false
$Script:bANSInstalled = $true
$Script:colImportAdapters = $null
$Script:EnabledSystemAdapters = $null

#Save off input parameters to be used in functions
$Script:Action = $Action
$Script:BDF = $BDF
$Script:ConfigPath = $ConfigPath
$Script:ScriptPath = $null


#------------------------ Messages ------------------------
$Script:Messages = @{
    Usage = @"
Intel(R) SaveRestore.ps1
Copyright 2014 Intel Corporation. All rights reserved.

  Usage: SaveRestore.ps1 -Action -ConfigPath -BDF
  Example: SaveRestore.ps1 –Action save –ConfigPath C:\*File Path*\config.txt
  -Action is required. Valid values are 'save' and 'restore.'
  -ConfigPath is optional. It specifies the path and file name of the main configuration save file. If not specified, it is the script path and default file name (Saved_Config.txt).
  NOTE: The Saved_StaticIP.txt file name does not change and is always saved and restored from the script path.
  -BDF is optional. If specified during a restore, it will restore settings to adapters using their bus/device/function.
  The default configuration file names are Saved_Config.txt and Saved_StaticIP.txt.
"@;

    # Performing a save/restore of configuration file at $Script:ConfigPath
    RunningSaveRestore        = "Performing a save/restore of configuration file at";

    SettingOEMValue           = "Setting OEM custom value.";

    # Restoring primary and secondary adapters on $Team.TeamName
    RestoringTeamAdapters     = "Restoring primary and secondary adapters on";

    # Restoring $Team.TeamName / $Vlan.VlanName
    Restoring                 = "Restoring"

    RestoringAdapterIP        = "Restoring adapter IP settings."
    RestoringTeamIP           = "Restoring team IP settings."
    RestoringVLANIP           = "Restoring VLAN IP settings."
    RestoringAdapterSettings  = "Restoring adapter settings."
    RestoringTeamSettings     = "Restoring team settings."
    RestoringVLANSettings     = "Restoring VLAN settings."
    RestoringAdapterPartition = "Restoring adapter partition settings."

    NICPartitioningDetected   = "Detected a device configured for NIC partitioning. We recommend you use the -BDF option when restoring."

    PerformingSave            = "Saving the configuration."
    PerformingUpgradeSave     = "Upgrading and saving the configuration."
    PerformingRestore         = "Restoring the configuration."
    PerformingUpgradeRestore  = "Upgrading and restoring the configuration."
    PerformingRemove          = "Removing the team and VLAN device configuration."

    # $Script:ConfigPath file already exists. Do you want to overwrite it (y/n)?
    FileExistsPrompt          = "file already exists. Do you want to overwrite it (y/n)?"

    #------------------------ Errors ------------------------

    # No objects found for $class
    NoObjectsFound            = "No objects found for"

    AdminRequired             = "The save/restore script requires administrative rights. Please log in as an Administrator and try again."
    PROSetRequired            = "The save/restore script requires Intel® PROSet to be installed. Please install Intel PROSet and try again."

    # Could not find the restore configuration file at $Script:ConfigPath. Please provide a path to the configuration file.
    ConfigNotFoundPart1       = "Could not find the restore configuration file at"
    ConfigNotFoundPart2       = "Please provide a valid path to the configuration file."

    # Configuration path $FilePath does not exist. Please provide a valid path.
    ConfigPathDoesntExist1    = "A directory in"
    ConfigPathDoesntExist2    = "does not exist. Please provide a valid path."

    ConfigFileNotSpecified    = "No configuration file name specified. Please provide a valid file name."

    # Unable to set ... on ...
    UnableToSetPart1          = "Unable to set"
    UnableToSetPart2          = "on"

    #----------------------- Warnings -----------------------
    UnableToFindDevice        = "Unable to find Device. Please verify configuration file matches your system configuration."
    UnableToFindANS           = "Intel(R) ANS is not present on the system. Any Team and VLAN information contained in the configuration file will not be restored."
}

#-----------------------------------------------------[Functions]------------------------------------------------------


# Set WMI lock
Function BeginApply
{
    # retrieve instance
        $Script:service = Get-WMiobject -class IAnet_netservice -namespace root\intelncs2

    # create [in] parameters
        $inparams = $Script:service.GetMethodParameters("BeginApply")

    # spawn management object
        $Script:options = new-object System.Management.InvokeMethodOptions

    # add context for this script
        $Script:options.Context.Add('SaveRestoreApply', $true)
        $Script:options.Context.Add('GET_EXTENSIONS', $true)
        $Script:options.Context.Add('GET_EXT_KEYS_ONLY', $false)
        $Script:options.Context.Add('IANet_PartialData', 512)

    # obtain lock
        $Script:lockobj = $Script:service.InvokeMethod("BeginApply", $inparams, $Script:options)

    # this is necessary to set the lock into a new object
        $Script:context = new-object System.Management.ManagementNamedValueCollection

    # adding a new single named value to the collection with the value of the lock objects client handle
        $Script:context.Add('ClientSetId', [int] $Script:lockobj.ClientSetHandle)
        $Script:context.Add('SaveRestoreApply', $true)
}

# Remove WMI lock
Function ReleaseLock
{
    $inparams = $Script:service.GetMethodParameters("Apply")
    $inparams.ClientSetHandle=[int]$Script:lockobj.ClientSetHandle
    $null = $Script:Service.InvokeMethod("Apply", $inparams, $null)
}

Function NewObject ($class)
{
    $classdef = New-Object System.Management.ManagementClass $class
    # the new object options
    $classdef.Options.UseAmendedQualifiers = $true
    $classdef.Options.Context = $Script:context

    return $classdef
}

Function DeleteObject ($object)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log $object
    # the delete options
    $deleteoptions = New-Object System.Management.DeleteOptions
    $deleteoptions.Context = $Script:context

    $object.Delete($deleteoptions)
}

Function SetSetting ($setting)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log "`$setting = $setting"
    # the put options
    $putoptions = New-Object System.Management.PutOptions($Script:context)
    $putoptions.UseAmendedQualifiers = $true

    #perform put
    $null = $setting.Put($putoptions)
}

# Get specified class object from WMI
Function GetObject($class)
{
    $querystring = [string] "SELECT * FROM $class"
    $query = New-Object System.Management.ObjectQuery($querystring)

    # the enumeration options
    $enumerate_option = New-Object System.Management.EnumerationOptions
    $enumerate_option.UseAmendedQualifiers = $true
    $enumerate_option.Context = $Script:context

    # setup scope
    $DMiXPath = New-Object System.Management.ManagementPath
    $DMiXPath.NamespacePath = "root\intelncs2"
    $scope = New-Object System.Management.ManagementScope($DMiXPath)
    $scope.Connect()

    # the searcher for the object
    $searcher = New-Object System.Management.ManagementObjectSearcher($scope, $query, $enumerate_option)
    $collection = $searcher.Get()

    # If the Get() above fails, it won't be caught until the return happens. So catch it and display an appropriate message.
    try
    {
        return $collection
    }
    catch
    {
        $ErrorMessage = [string]$Messages.NoObjectsFound
        Write-Error "$ErrorMessage $class"
        Write-Log "$ErrorMessage $class"
        exit
    }
}

# Get associated objects given the object path and where condition
Function GetAssociated($path, $where)
{
    $querystring = [string] "ASSOCIATORS OF {$path} WHERE $where"

    $query = New-Object System.Management.ObjectQuery($querystring)

    # the enumeration options
    $enumerate_option = New-Object System.Management.EnumerationOptions
    $enumerate_option.UseAmendedQualifiers = $true
    $enumerate_option.Context = $Script:context

    # setup scope
    $DMiXPath = New-Object System.Management.ManagementPath
    $DMiXPath.NamespacePath = "root\intelncs2"
    $scope = New-Object System.Management.ManagementScope($DMiXPath)
    $scope.Connect()

    # the searcher for the object
    $searcher = New-Object System.Management.ManagementObjectSearcher($scope, $query, $enumerate_option)
    $collection = $searcher.Get()

    return $collection
}

# Invoke a method given it's name, class and parameters
Function InvokeMethod($Class,$Method,$ColParameterName,$ColParameterValue)
{
    $Invoke_option = New-Object System.Management.InvokeMethodOptions
    $Invoke_option.Context = $Script:context

    $params = $null
    if ($null -ne $ColParameterName)
    {
        $params = $Class.psbase.GetMethodParameters($Method)
        if ($ColParameterName.count -gt 1)
        {
            for($i=0; $i -lt $ColParameterName.count; $i++)
            {
                $params.psbase.SetPropertyValue($ColParameterName[$i],$ColParameterValue[$i])
            }
        }
        else
        {
            $params.psbase.SetPropertyValue($ColParameterName,$ColParameterValue)
        }
    }

    return $Class.psbase.InvokeMethod($Method,$params,$Invoke_option)
}

# Function used to write objects to config file.
Function WriteObjectToConfigFile($NewObjects,$Properties)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log "`$NewObjects = $NewObjects"
    Write-Log "`$Properties = $Properties"
    if ($null -ne $NewObjects)
    {
        $ExportObject = $NewObjects | Select-Object $Properties | ConvertTo-Csv -NoTypeInformation
        $ExportObject = $ExportObject[1..$ExportObject.Count]
        WriteTextToConfigFile $ExportObject
    }
}

# Function used to write text to config file.
Function WriteTextToConfigFile($NewText)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log "`$NewText = $NewText"
    $NewText | Out-File $Script:ConfigPath -Encoding utf8 -Append
}

# Function used to write objects to the IP config file.
Function WriteObjectToIPFile($NewObjects,$Properties)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log "`$NewObjects = $NewObjects"
    Write-Log "`$Properties = $Properties"
    if ($null -ne $NewObjects)
    {
        $ExportObject = $NewObjects | Select-Object $Properties | ConvertTo-Csv -NoTypeInformation
        $ExportObject = $ExportObject[1..$ExportObject.Count]
        WriteTextToIPFile $ExportObject
    }
}

# Function used to write text to IP config file.
Function WriteTextToIPFile($NewText)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log "`$NewText = $NewText"
    $NewText | Out-File $Script:ScriptPath\Saved_StaticIP.txt -Encoding utf8 -Append
}

# Function used to read objects from config file given a file section.
Function ReadFromConfigFile($FileSection)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log "`$FileSection = $FileSection"
    $FileObjects=@()
    $FileObjects = Get-Content $Script:ConfigPath

    switch($FileSection)
    {
        "Adapters"
        {
            #Find the section for adapters and add 1 to not include AdaptersStart identifier
            $CsvObjectStart = [array]::indexof($FileObjects,"AdaptersStart") + 1
            #Find the end of the section and remove 1 to not include AdaptersEnd identifier
            $CsvEnd = [array]::indexof($FileObjects,"AdaptersEnd") - 1

            $colProperty = "Name","OriginalDisplayName","PermanentAddress","PCIDeviceID","SlotID"
            break
        }
        "AdapterSettings"
        {
            #Find the section for adapters and add 1 to not include AdapterSettingsStart identifier
            $CsvObjectStart = [array]::indexof($FileObjects,"AdapterSettingsStart") + 1
            #Find the end of the section and remove 1 to not include AdapterSettingsEnd identifier
            $CsvEnd = [array]::indexof($FileObjects,"AdapterSettingsEnd") - 1

            $colProperty = "Name","Caption","CurrentValue","CurrentValues","PermanentAddress","PCIDeviceID","SlotID"
            break
        }
        "OEMSetting"
        {
            #Find the section for adapters and add 1 to not include OEMSettingStart identifier
            $CsvObjectStart = [array]::indexof($FileObjects,"OEMSettingStart") + 1
            #Find the end of the section and remove 1 to not include OEMSettingEnd identifier
            $CsvEnd = [array]::indexof($FileObjects,"OEMSettingEnd") - 1

            $colProperty = "OEMCustomizeable"
            break
        }
        "PMSettings"
        {
            #Find the section for adapters and add 1 to not include PMSettingsStart identifier
            $CsvObjectStart = [array]::indexof($FileObjects,"PMSettingsStart") + 1
            #Find the end of the section and remove 1 to not include PMSettingsEnd identifier
            $CsvEnd = [array]::indexof($FileObjects,"PMSettingsEnd") - 1

            $colProperty = "Name","Caption","CurrentValue","CurrentValues","PermanentAddress","PCIDeviceID","SlotID"
            break
        }
        "Teams"
        {
            #Find the section for adapters and add 1 to not include TeamsStart identifier
            $CsvObjectStart = [array]::indexof($FileObjects,"TeamsStart") + 1
            #Find the end of the section and remove 1 to not include TeamsEnd identifier
            $CsvEnd = [array]::indexof($FileObjects,"TeamsEnd") - 1

            $colProperty = "TeamName","TeamMembers","TeamMode","PrimaryAdapter","SecondaryAdapter"
            break
        }
        "TeamSettings"
        {
            #Find the section for adapters and add 1 to not include TeamSettingsStart identifier
            $CsvObjectStart = [array]::indexof($FileObjects,"TeamSettingsStart") + 1
            #Find the end of the section and remove 1 to not include TeamSettingsEnd identifier
            $CsvEnd = [array]::indexof($FileObjects,"TeamSettingsEnd") - 1

            $colProperty = "TeamName","Caption","CurrentValue","CurrentValues"
            break
        }
        "Vlans"
        {
            #Find the section for adapters and add 1 to not include VlansStart identifier
            $CsvObjectStart = [array]::indexof($FileObjects,"VlansStart") + 1
            #Find the end of the section and remove 1 to not include VlansEnd identifier
            $CsvEnd = [array]::indexof($FileObjects,"VlansEnd") - 1

            $colProperty = "ParentName","VLANID","VLANNAME","ParentPermanentAddress","ParentPCIDeviceID","ParentSlotID"
            break
        }
        "VlanSettings"
        {
            #Find the section for adapters and add 1 to not include VlanSettingsStart identifier
            $CsvObjectStart = [array]::indexof($FileObjects,"VlanSettingsStart") + 1
            #Find the end of the section and remove 1 to not include VlanSettingsEnd identifier
            $CsvEnd = [array]::indexof($FileObjects,"VlanSettingsEnd") - 1

            $colProperty = "ParentName","Name","VLANID","Caption","CurrentValue","ParentPermanentAddress","ParentPCIDeviceID","ParentSlotID"
            break
        }
        "NICPARTSettings"
        {
            #Find the section for adapters and add 1 to not include NICPARTSettingsStart identifier
            $CsvObjectStart = [array]::indexof($FileObjects,"NICPARTSettingsStart") + 1
            #Find the end of the section and remove 1 to not include NICPARTSettingsEnd identifier
            $CsvEnd = [array]::indexof($FileObjects,"NICPARTSettingsEnd") - 1

            $colProperty = "Name","PartitionNumber","Identifier","MinBWPercent","MaxBWPercent"
            break
        }
        Default
        {
            return
        }
    }

    #If no items were found in the provided section return nothing
    if ($CsvEnd -lt $CsvObjectStart)
    {
        return

    }
    #Else return only the provided section and convert the text into objects
    else
    {
        return $FileObjects[$CsvObjectStart..$CsvEnd] | ConvertFrom-Csv -Header $colProperty
    }
}

# Function used to read objects from config file.
Function ReadFromIPFile($FileSection)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log "`$FileSection = $FileSection"
    $FileObjects=@()
    $FileObjects = Get-Content $Script:ScriptPath\Saved_StaticIP.txt

    switch($FileSection)
    {
        "AdapterIPSettings"
        {
            #Find the section for adapters and add 1 to not include AdapterIPSettingsStart identifier
            $CsvObjectStart = [array]::indexof($FileObjects,"AdapterIPSettingsStart") + 1
            #Find the end of the section and remove 1 to not include AdapterIPSettingsEnd identifier
            $CsvEnd = [array]::indexof($FileObjects,"AdapterIPSettingsEnd") - 1

            $colProperty = "Name","Caption","CurrentValue","CurrentValues","PermanentAddress","PCIDeviceID","SlotID"
            break
        }
        "TeamIPSettings"
        {
            #Find the section for adapters and add 1 to not include TeamIPSettingsStart identifier
            $CsvObjectStart = [array]::indexof($FileObjects,"TeamIPSettingsStart") + 1
            #Find the end of the section and remove 1 to not include TeamIPSettingsEnd identifier
            $CsvEnd = [array]::indexof($FileObjects,"TeamIPSettingsEnd") - 1

            $colProperty = "TeamName","Caption","CurrentValue","CurrentValues"
            break
        }
        "VlanIPSettings"
        {
            #Find the section for adapters and add 1 to not include VlanIPSettingsStart identifier
            $CsvObjectStart = [array]::indexof($FileObjects,"VlanIPSettingsStart") + 1
            #Find the end of the section and remove 1 to not include VlanIPSettingsEnd identifier
            $CsvEnd = [array]::indexof($FileObjects,"VlanIPSettingsEnd") - 1

            $colProperty = "ParentName","VLANID","Caption","CurrentValue","CurrentValues","ParentPermanentAddress","ParentPCIDeviceID","ParentSlotID"
            break
        }
        Default
        {
            return
        }
    }

    # If no items were found in the provided section return nothing
    if ($CsvEnd -lt $CsvObjectStart)
    {
        return
    }
    # Else return only the provided section and convert the text into objects
    else
    {
        return $FileObjects[$CsvObjectStart..$CsvEnd] | ConvertFrom-Csv -Header $colProperty
    }
}

# Add the parent adapter identifiers to the setting objects
Function AddParentIDs($Setting)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log "`$Setting = $Setting"
    #Add the PermanentAddress, PCIDeviceID and SlotID to the settings object
    $SettingAdapter = $Script:Adapters | Where-Object {$_.DeviceID -eq $Setting.ParentId}
    if ($SettingAdapter)
    {
        $Setting | Add-Member -Name "PermanentAddress" -Value  $SettingAdapter.PermanentAddress -MemberType NoteProperty
        $Setting | Add-Member -Name "PCIDeviceID" -Value  $SettingAdapter.PCIDeviceID -MemberType NoteProperty
        $AdapterSlotID =  $SettingAdapter.SlotID.split(":")[0..2] #return only the first three elements in array
        $AdapterBDF = [string]::Join(':',$AdapterSlotID)
        $Setting | Add-Member -Name "SlotID" -Value  $AdapterBDF -MemberType NoteProperty
    }
}

Function PrintUsage
{
    Write-Host $Messages.Usage
    exit
}

Function CheckForAdminRights
{
    $winIdent = [Security.Principal.WindowsIdentity]::GetCurrent()
    $WinPrinc = [Security.Principal.WindowsPrincipal] $winIdent
    $AdminId  = [Security.Principal.WindowsBuiltInRole] "Administrator"
    if (-Not $WinPrinc.IsInRole($AdminId))
    {
        Write-Error $Messages.AdminRequired
        exit
    }
}

Function CheckForDMiXInstall
{
    Write-Log $MyInvocation.MyCommand
    $DMiXInstall = Get-ItemProperty -Path HKLM:\SOFTWARE\Intel\Network_Services\DMIX -Name InstalledDMIX -ErrorAction SilentlyContinue

    if ((!$DMiXInstall) -or ($DMiXInstall.InstalledDMIX -ne 1))
    {
        Write-Error $Messages.PROSetRequired
        exit
    }
}

Function CheckForANSInstall
{
    Write-Log $MyInvocation.MyCommand
    $ANSInstall = Get-ItemProperty -Path HKLM:\SOFTWARE\Intel\Network_Services\DMIX -Name InstalledDMIX_ANS -ErrorAction SilentlyContinue

    if ((!$ANSInstall) -or ($ANSInstall.InstalledDMIX_ANS -ne 1))
    {
        $Script:bANSInstalled = $false
    }
}

Function CheckIfConfigFileExists
{
    Write-Log $MyInvocation.MyCommand
    if (!(Test-Path $Script:ConfigPath -PathType Leaf))
    {
        $ErrorMessagePart1 = [string]$Messages.ConfigNotFoundPart1
        $ErrorMessagePart2 = [string]$Messages.ConfigNotFoundPart2

        Write-Error  "$ErrorMessagePart1 $Script:ConfigPath. $ErrorMessagePart2"
        exit
    }
}

Function CheckIfConfigShouldBeOverwritten
{
    Write-Log $MyInvocation.MyCommand
    if (!(Test-Path $Script:ConfigPath.ToString() -PathType Container))
    {
        $FileExistsString = [string]$Messages.FileExistsPrompt
        $confirmation = Read-Host -Prompt "$Script:ConfigPath $FileExistsString"
        if ($confirmation -notmatch "[yY]")
        {
            exit
        }
    }
}

Function Write-Log($string)
{
    $timestamp = "[$(Get-Date -Format G)] "
    $string = $timestamp + $string
    Write-Verbose -Message $string
}

# Sets a global variable for the configuration file paths to be saved/restored from.
Function SetupSaveRestoreLocation
{
    Write-Log $MyInvocation.MyCommand
    $Script:ScriptPath = Get-Location

    # If the user did not specify a path, use the scripts path
    if ($Script:bUpSave -eq $true -or $Script:bUpRestore -eq $true)
    {
        $Script:ConfigPath = Get-Location
        $Script:ConfigPath = $Script:ConfigPath + "\Upgrade_Saved_Config.txt"
    }
    elseif (!$Script:ConfigPath)
    {
        $Script:ConfigPath = Get-Location
        $Script:ConfigPath = $Script:ConfigPath + "\Saved_Config.txt"

        #Check if the file exist during a save
        if ($Script:bSave -eq $true -and (Test-Path $Script:ConfigPath -PathType Leaf))
        {
            CheckIfConfigShouldBeOverwritten
        }
    }
    else
    {
        #Only prompt to overwrite if we are doing save and file exists
        if (($Script:bSave -eq $true -or $Script:bUpSave -eq $true) -and (Test-Path $Script:ConfigPath -PathType Leaf))
        {
            CheckIfConfigShouldBeOverwritten
        }

        #Current dir is default
        $FilePath = "."
        #Find position of the last backslash before the filename
        $FileNamePos = $Script:ConfigPath.ToString().LastIndexOf("\")
        #pos = -1 means ConfigPath contains just the file name - use current dir in this case
        if ($FileNamePos -ne -1)
        {
            #Separate the filename from the path to verify path exists
            $FilePath = $Script:ConfigPath.ToString().Substring(0, $FileNamePos)
        }

        #Check that config path exists
        if (!(Test-Path $FilePath -PathType Container))
        {
            $ErrorMessagePart1 = [string]$Messages.ConfigPathDoesntExist1
            $ErrorMessagePart2 = [string]$Messages.ConfigPathDoesntExist2
            Write-Error "$ErrorMessagePart1 $FilePath $ErrorMessagePart2"
            exit
        }

        #Check the input of the config file when Saving
        if ($Script:bSave -eq $true -or $Script:bUpSave -eq $true)
        {
            Try
            {
                Out-File $Script:ConfigPath  -ErrorAction Stop
            }
            Catch
            {
                Write-Error $Messages.ConfigFileNotSpecified
                Write-Log $Messages.ConfigFileNotSpecified
                exit
            }
        }
        #Check the input of the config file when Restoring
        elseif (!(Test-Path $Script:ConfigPath))
        {
            Write-Error $Messages.ConfigFileNotSpecified
            exit
        }
    }

    Write-Host $Messages.RunningSaveRestore $Script:ConfigPath
}

# Get the present and enabled adapters on the system.
Function GetAdaptersOnSystem
{
    Write-Log $MyInvocation.MyCommand
    #Only store adapters that have a status of 3 (which means enabled)
    $Script:EnabledSystemAdapters = GetObject "IANet_PhysicalEthernetAdapter" | Where-Object {($_.StatusInfo -eq 3)}  | Sort-Object -Property Name
    #Only store adapters that support DMiX or Extended DMiX capability or if it is intel vendor capable and this is an upgrade.
    $Script:Adapters = $Script:EnabledSystemAdapters | Where-Object {($_.Capabilities -eq 73) -or ($_.Capabilities -eq 74) -or (($_.Capabilities -eq 47) -and (($bUpSave -eq $true) -or ($bUpRestore -eq $true)))}
    #Save Teamable adapters to be referenced for teams
}

# Remove any present configuration files
Function RemoveOldFiles
{
    Write-Log $MyInvocation.MyCommand
    # check if the file exists before trying to remove it
    if (Test-Path $Script:ConfigPath -PathType Leaf)
    {
        Remove-Item $Script:ConfigPath
    }
    if (Test-Path $Script:ScriptPath\Saved_StaticIP.txt)
    {
        Remove-Item $Script:ScriptPath\Saved_StaticIP.txt
    }
}

# Save adapters on system to the configuration file
Function SaveAdapters
{
    Write-Log $MyInvocation.MyCommand
    WriteTextToConfigFile "AdaptersStart"

    $colProperty = "Name","OriginalDisplayName","PermanentAddress","PCIDeviceID","SlotID"
    WriteObjectToConfigFile $Script:EnabledSystemAdapters $colProperty

    WriteTextToConfigFile "AdaptersEnd"
    WriteTextToConfigFile ""
}

# Save OEM customization value
Function SaveOEMCustomizeableSetting
{
    Write-Log $MyInvocation.MyCommand
    WriteTextToConfigFile "OEMSettingStart"
    $OEMSetting = Get-ItemProperty -Path HKLM:\SOFTWARE\Intel\Network_Services\NCS2 -Name OEMCustomizeable -ErrorAction SilentlyContinue

    # If a value is present save it to the config file
    if ($null -ne $OEMSetting)
    {
        WriteObjectToConfigFile $OEMSetting OEMCustomizeable
    }

    WriteTextToConfigFile "OEMSettingEnd"
    WriteTextToConfigFile ""
}

# Save any legacy power management settings
Function SavePowerManagementSettings
{
    Write-Log $MyInvocation.MyCommand
    $colProperty = "Name","Caption","CurrentValue",@{expression={$_.CurrentValues -join ","};label="CurrentValues"},"PermanentAddress","PCIDeviceID","SlotID"

    WriteTextToConfigFile "PMSettingsStart"

    foreach ($Adapter in $Script:Adapters)
    {
        try
        {
            #Get the Power Management settings for adapter
            $PowerSettingValues = InvokeMethod $Adapter GetPowerUsageOptions $null $null
            $PowerSettingProperties = @{Name = $Adapter.Name;
                                        CurrentValue = $PowerSettingValues.AutoPowerSaveModeEnabled;
                                        CurrentValues = {};
                                        PermanentAddress = $Adapter.PermanentAddress;
                                        PCIDeviceID = $Adapter.PCIDeviceID;
                                        SlotID = $Adapter.SlotID}

            # Check each Power Management setting to see if it NULL before trying to save it to the config file
            # if there is a value, create a custom object and save it to the config file
            if ($null -ne $PowerSettingValues.AutoPowerSaveModeEnabled)
            {
                $Setting = New-Object PSObject -Property $PowerSettingProperties
                $Setting | Add-Member -Name "Caption" -Value  "AutoPowerSaveModeEnabled" -MemberType NoteProperty
                WriteObjectToConfigFile $Setting $colProperty
            }
            if ($null -ne $PowerSettingValues.ReduceSpeedOnPowerDown)
            {
                $Setting = New-Object PSObject -Property $PowerSettingProperties
                $Setting | Add-Member -Name "Caption" -Value  "ReduceSpeedOnPowerDown" -MemberType NoteProperty
                WriteObjectToConfigFile $Setting $colProperty
            }
            if ($null -ne $PowerSettingValues.SmartPowerDown)
            {
                $Setting = New-Object PSObject -Property $PowerSettingProperties
                $Setting | Add-Member -Name "Caption" -Value  "SmartPowerDown" -MemberType NoteProperty
                WriteObjectToConfigFile $Setting $colProperty
            }
            if ($null -ne $PowerSettingValues.SavePowerNowEnabled)
            {
                $Setting = New-Object PSObject -Property $PowerSettingProperties
                $Setting | Add-Member -Name "Caption" -Value  "SavePowerNowEnabled" -MemberType NoteProperty
                WriteObjectToConfigFile $Setting $colProperty
            }
            if ($null -ne $PowerSettingValues.EnhancedASPMPowerSaver)
            {
                $Setting = New-Object PSObject -Property $PowerSettingProperties
                $Setting | Add-Member -Name "Caption" -Value  "EnhancedASPMPowerSaver" -MemberType NoteProperty
                WriteObjectToConfigFile $Setting $colProperty
            }
        }
        catch
        {
            Write-Log "Unable to Save Power Management Settings"
        }
    }

    WriteTextToConfigFile "PMSettingsEnd"
    WriteTextToConfigFile ""
}

# Some settings need to be saved in a certain order, this function stops them from being saved now so they can be saved later by returning
# whether it should be saved immidiately or later (using true or false).
Function SaveAdapterSettingLater($Setting, $bIPSetting)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log "`$Setting = $Setting"
    Write-Log "`$bIPSetting = $bIPSetting"
    $bRet = $true

    # Don't save the settings now if it is an IP, DCB, performance profile, SRIOV, VMQueues, or NUMVF setting
    if (($bIPSetting -eq $false) -and
        ($Setting.GroupId -ne 12 ) -and
        ($Setting.GroupId -ne 8 ) -and
        ($Setting.Caption -ne "PerformanceProfile") -and
        ($Setting.Caption -ne "*FlowControl") -and
        ($Setting.Caption -ne "*PriorityVLANTag") -and
        ($Setting.Caption -ne "*SRIOV") -and
        ($Setting.Caption -ne "VMQueues") -and
        ($Setting.Caption -ne "*NumVFs"))
    {
        $bRet = $false
    }

    return $bRet
}

# Check if the given setting is an IP setting and save it in the IP config file and return if it is an IP setting (true or false).
# Depending on the device different properties of the object need to be saved.
Function SaveIPSetting($Setting,$DeviceType,$EnabledDHCP,$bSaveIPv6Settings)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log "`$DeviceType = $DeviceType"
    Write-Log "`$Setting = $Setting"
    Write-Log "`$bEnabledDHCP = $EnabledDHCP"
    Write-Log "`$bSaveIPv6Settings = $bSaveIPv6Settings"
    $bIPSetting = $false
    $bSaveIpSetting = $false

    # Check if the passed in setting is one of these IP settings.
    # Some IP settings need DHCP enabled to restore.
    switch($Setting.Caption)
    {
        "IPAddress"
        {
            $bIPSetting = $true
            if ($EnabledDHCP -eq 0)
            {
                $bSaveIpSetting = $true
            }
        }
        "IPv4Address"
        {
            $bIPSetting=$true
            if ($EnabledDHCP -eq 0)
            {
                $bSaveIpSetting = $true
            }
        }
        "IPv6Address"
        {
            $bIPSetting=$true
            if ($bSaveIPv6Settings -eq $true)
            {
                $bSaveIpSetting = $true
            }
        }
        "PrefixLength"
        {
            $bIPSetting=$true
            if ($bSaveIPv6Settings -eq $true)
            {
                $bSaveIpSetting = $true
            }
        }
        "SubnetMask"
        {
            $bIPSetting=$true
            if ($EnabledDHCP -eq 0)
            {
                $bSaveIpSetting = $true
            }
        }
        "DefaultGateway"
        {
            $bIPSetting=$true
            if ($EnabledDHCP -eq 0)
            {
                $bSaveIpSetting = $true
            }
        }
        "DefaultGatewayIPv6"
        {
            $bIPSetting=$true
            if ($bSaveIPv6Settings -eq $true)
            {
                $bSaveIpSetting = $true
            }
        }
        "NameServer"
        {
            $bIPSetting=$true
            if($EnabledDHCP -eq 0)
            {
                $bSaveIpSetting = $true
            }
        }
        "NameServerList"
        {
            $bIPSetting=$true
            $bSaveIpSetting = $true
        }
        "NetbiosOptions"
        {
            $bIPSetting=$true
            $bSaveIpSetting = $true
        }
    }

    # Save IP settings with different properties depending on the device type
    if ($bSaveIpSetting -eq $true -and $DeviceType -eq "Adapter")
    {
        $colProperty = "Name","Caption","CurrentValue",@{expression={$_.CurrentValues -join ","};label="CurrentValues"},"PermanentAddress","PCIDeviceID","SlotID"
        WriteObjectToIPFile $Setting $colProperty
    }
    elseif ($bSaveIpSetting -eq $true -and $DeviceType -eq "Team")
    {
        $colProperty = "TeamName","Caption","CurrentValue",@{expression={$_.CurrentValues -join ","};label="CurrentValues"}
        WriteObjectToIPFile $Setting $colProperty
    }
    elseif ($bSaveIpSetting -eq $true -and $DeviceType -eq "Vlan")
    {
        $colProperty = "ParentName","VLANID","Caption","CurrentValue",@{expression={$_.CurrentValues -join ","};label="CurrentValues"},"ParentPermanentAddress","ParentPCIDeviceID","ParentSlotID"
        WriteObjectToIPFile $Setting $colProperty
    }

    return $bIPSetting
}

# Save the adapter settings
Function SaveAdapterSettings()
{
    Write-Log $MyInvocation.MyCommand
    $colProperty = "Name","Caption","CurrentValue",@{expression={$_.CurrentValues -join ","};label="CurrentValues"},"PermanentAddress","PCIDeviceID","SlotID"

    # Save power management settings
    SavePowerManagementSettings

    WriteTextToConfigFile "AdapterSettingsStart"
    WriteTextToIPFile "AdapterIPSettingsStart"

    foreach ($Adapter in $Script:Adapters)
    {
        # Get the settings associated with the adapter
        $colSettings = GetAssociated $Adapter.path.path "ResultClass = IANet_AdapterSetting"
        $colSettings = $colSettings  | Sort-Object -Property Name,Caption

        # Check status of EnablDHCP for IP settings later
        $EnableDHCPCol = $colSettings | Where-Object {($_.Caption -eq "EnableDHCP")} | Select-Object $colProperty
        $EnableDHCPv6Col = $colSettings | Where-Object {($_.Caption -eq "EnableDHCPv6")} | Select-Object $colProperty
        $ManualIPv6Col = $colSettings | Where-Object {($_.Caption -eq "ManualIPv6")} | Select-Object $colProperty

        #Get the DHCP enable value for the specific adapter
        $EnableDHCP = $EnableDHCPCol | Where-Object {($_.Name -eq $Adapter.Name)}

        # Get the DHCPv6 enable value, and the ManualIPv6 value
        # EnableDHCPv6 is 0 (disabled) only if the "Managed Address Configuration Flag" and the "Other Stateful Configuration Flag" are both 0
        # (see https://blogs.technet.microsoft.com/teamdhcp/2009/03/03/dhcpv6-understanding-of-address-configuration-in-automatic-mode-and-installation-of-dhcpv6-server/)
        # ManualIPv6 is 1 (manual) only if the PrefixOrigin and SuffixOrigin of any IPv6 address are both "Manual" - other values indicate autoconfiguration
        $EnableDHCPv6 = $EnableDHCPv6Col | Where-Object {($_.Name -eq $Adapter.Name)}
        $ManualIPv6 = $ManualIPv6Col | Where-Object {($_.Name -eq $Adapter.Name)}

        $bSaveIPv6Settings = $false
        #if EnableDHCPv6 = 0 and ManualIPv6 = 1, we want to save IPv6 addresses and settings. Otherwise, all IPv6 addresses were autoconfigured and we don't need to save/restore them
        if($EnableDHCPv6.CurrentValue -eq 0 -and $ManualIPv6.CurrentValue -eq 1)
        {
            $bSaveIPv6Settings = $true
        }

        foreach ($Setting in $colSettings)
        {
            AddParentIDs $Setting

            #check if setting is an IP setting save them in the IP config file instead of the Saved_Config file
            $bIPSetting = SaveIPSetting $Setting "Adapter" $EnableDHCP.CurrentValue $bSaveIPv6Settings

            #Check to see if the setting should be saved later
            $bRet = SaveAdapterSettingLater $Setting $bIPSetting
            if ($bRet -eq $false)
            {
                WriteObjectToConfigFile $Setting $colProperty
            }
        }

        # Check if DCB is being updated and if so, don't save the settings so the default values are restored
        $RestoreDCB = $true
        $FCoEUpdate = Get-ItemProperty -Path HKLM:\SOFTWARE\Intel\Prounstl -Name DCB_Update_FCoE  -ErrorAction SilentlyContinue
        if ($null -ne $FCoEUpdate)
        {
            if ($FCoEUpdate.DCB_Update_FCoE -eq 1)
            {
                #FCoE is changing don't save settings
                $RestoreDCB = $false
            }
        }
        $iSCSIUpdate = Get-ItemProperty -Path HKLM:\SOFTWARE\Intel\Prounstl -Name DCB_Update_iSCSI  -ErrorAction SilentlyContinue
        if ($null -ne $iSCSIUpdate)
        {
            if ($iSCSIUpdate.DCB_Update_iSCSI -eq 1)
            {
                #iSCSI is changing don't save settings
                $RestoreDCB = $false
            }
        }

        #Save *SRIOV after *VMQ
        $SRIOVSetting = $colSettings | Where-Object {($_.Caption -eq "*SRIOV")}
        WriteObjectToConfigFile $SRIOVSetting $colProperty

        #Save DCB Settings if this is not an upgrade or if it is an upgrade, and we are modifying DCB
        if (($Script:bUpSave -eq $false) -or ($RestoreDCB -eq $true))
        {
            $colDCBSettings = $colSettings | Where-Object {($_.GroupId -eq 12) -or ($_.GroupId -eq 8)}
            WriteObjectToConfigFile $colDCBSettings $colProperty
        }

        #Save the performance profile
        $ProfileSetting = $colSettings | Where-Object {($_.Caption -eq "PerformanceProfile")}
        WriteObjectToConfigFile $ProfileSetting $colProperty

        #Save the FlowControl after performance profile and DCB
        $FlowControlSetting = $colSettings | Where-Object {($_.Caption -eq "*FlowControl")}
        WriteObjectToConfigFile $FlowControlSetting $colProperty

        #Save the PriorityVLANTag after performance profile and DCB
        $PriorityVLANTagSetting = $colSettings | Where-Object {($_.Caption -eq "*PriorityVLANTag")}
        WriteObjectToConfigFile $PriorityVLANTagSetting $colProperty

        #Save VMQueues and *NUMVFs last
        $VMQQueuesSetting = $colSettings | Where-Object {($_.Caption -eq "VMQueues")}
        WriteObjectToConfigFile $VMQQueuesSetting $colProperty
        $NumVFsSetting = $colSettings | Where-Object {($_.Caption -eq "*NumVFs")}
        WriteObjectToConfigFile $NumVFsSetting $colProperty
    }

    WriteTextToConfigFile "AdapterSettingsEnd"
    WriteTextToConfigFile ""

    WriteTextToIPFile "AdapterIPSettingsEnd"
    WriteTextToIPFile ""
}

#Save team information
Function SaveTeams
{
    Write-Log $MyInvocation.MyCommand
    #Get current teams on system
    $colProperty = "TeamName","TeamMembers","TeamMode","PrimaryAdapter","SecondaryAdapter"
    $colItems = GetObject "IANet_TeamOfAdapters" | Select-Object $colProperty | Sort-Object -Property TeamName


    WriteTextToConfigFile "TeamsStart"

    # if there are teams on system save the team information to the config file.
    if ($colItems)
    {
        # convert Member arrays into a "single" value to save
        foreach ($item in $colItems)
        {
            $TeamMembers = $item.TeamMembers
            $item.TeamMembers = @()
            for (($i = 0); $i -lt $TeamMembers.Count; $i++)
            {
                $adapter = GetObject IANet_PhysicalEthernetAdapter | Where-Object {($_.OriginalDisplayName -eq $TeamMembers[$i])}
                $PermanentAddress = $adapter | Select-Object -ExpandProperty "PermanentAddress"
                $PCIDeviceID = $adapter | Select-Object -ExpandProperty "PCIDeviceID"
                $SlotID = $adapter | Select-Object -ExpandProperty "SlotID"
                $item.TeamMembers +=  $TeamMembers[$i] + "," + $PermanentAddress + "," + $PCIDeviceID + "," + $SlotID
            }
            $item.TeamMembers = $item.TeamMembers -join '|'
        }

        WriteObjectToConfigFile $colItems *
    }

    WriteTextToConfigFile "TeamsEnd"
    WriteTextToConfigFile ""
}

# Save team settings
Function SaveTeamSettings
{
    Write-Log $MyInvocation.MyCommand
    # Get the current team settings
    $colProperty = "TeamName","Caption","CurrentValue",@{expression={$_.CurrentValues -join ","};label="CurrentValues"}
    $colSettings = GetObject "IANet_TeamSetting" | Sort-Object -Property Name,Caption

    WriteTextToConfigFile "TeamSettingsStart"
    WriteTextToIPFile "TeamIPSettingsStart"

    # Check status of EnablDHCP for IP settings later
    $EnableDHCPCol = $colSettings | Where-Object {($_.Caption -eq "EnableDHCP")} | Select-Object $colProperty
    $EnableDHCPv6Col = $colSettings | Where-Object {($_.Caption -eq "EnableDHCPv6")} | Select-Object $colProperty
    $ManualIPv6Col = $colSettings | Where-Object {($_.Caption -eq "ManualIPv6")} | Select-Object $colProperty

    foreach ($Setting in $colSettings)
    {
        # Get the DHCP enable value for the specific Team
        $EnableDHCP = $EnableDHCPCol | Where-Object {($_.TeamName -eq $Setting.Name)}

        # Get the DHCPv6 enable value, and the ManualIPv6 value
        # EnableDHCPv6 is 0 (disabled) only if the "Managed Address Configuration Flag" and the "Other Stateful Configuration Flag" are both 0
        # (see https://blogs.technet.microsoft.com/teamdhcp/2009/03/03/dhcpv6-understanding-of-address-configuration-in-automatic-mode-and-installation-of-dhcpv6-server/)
        # ManualIPv6 is 1 (manual) only if the PrefixOrigin and SuffixOrigin of any IPv6 address are both "Manual" - other values indicate autoconfiguration
        $EnableDHCPv6 = $EnableDHCPv6Col | Where-Object {($_.TeamName -eq $Setting.Name)}
        $ManualIPv6 = $ManualIPv6Col | Where-Object {($_.TeamName -eq $Setting.Name)}

        $bSaveIPv6Settings = $false
        # if EnableDHCPv6 = 0 and ManualIPv6 = 1, we want to save IPv6 addresses and settings. Otherwise, all IPv6 addresses were autoconfigured and we don't need to save/restore them
        if ($EnableDHCPv6.CurrentValue -eq 0 -and $ManualIPv6.CurrentValue -eq 1)
        {
            $bSaveIPv6Settings = $true
        }

        # Save the IP Settings in the IP config file
        $bIPSetting = SaveIPSetting $Setting "Team" $EnableDHCP.CurrentValue $bSaveIPv6Settings

        if ($bIPSetting -eq $false)
        {
            WriteObjectToConfigFile $Setting $colProperty
        }
    }

    WriteTextToConfigFile "TeamSettingsEnd"
    WriteTextToConfigFile ""

    WriteTextToIPFile "TeamIPSettingsEnd"
    WriteTextToIPFile ""
}

# Save vlan information
Function SaveVlans
{
    Write-Log $MyInvocation.MyCommand
    #Get the vlans on the system
    $colProperty = "ParentName", "VLANID", "VLANNAME", "ParentPermanentAddress", "ParentPCIDeviceID", "ParentSlotID"
    $colVlans = GetObject "IANet_VLAN" | Sort-Object -Property Parent,VLANID


    WriteTextToConfigFile "VlansStart"

    # Save untagged vlan last if there are vlans on the system
    if ($colVlans)
    {
        $colTaggedVlans = $colVlans | Where-Object {($_.VLANID -ne 0)}
        WriteObjectToConfigFile $colTaggedVlans $colProperty
        $colUnTaggedVlans = $colVlans | Where-Object {($_.VLANID -eq 0)}
        WriteObjectToConfigFile $colUnTaggedVlans $colProperty
    }

    WriteTextToConfigFile "VlansEnd"
    WriteTextToConfigFile ""
}

# Save vlan settings
Function SaveVlanSettings
{
    Write-Log $MyInvocation.MyCommand
    #Get vlan settings on system
    $colProperty = "ParentName","Name","VLANID","Caption","CurrentValue","ParentPermanentAddress","ParentPCIDeviceID","ParentSlotID"
    $colSettings = GetObject "IANet_VLANSetting" | Sort-Object -Property Name,Caption

    WriteTextToConfigFile "VlanSettingsStart"
    WriteTextToIPFile "VlanIPSettingsStart"

    # Check status of EnablDHCP for IP settings later
    $EnableDHCPCol = $colSettings | Where-Object {($_.Caption -eq "EnableDHCP")} | Select-Object $colProperty
    $EnableDHCPv6Col = $colSettings | Where-Object {($_.Caption -eq "EnableDHCPv6")} | Select-Object $colProperty
    $ManualIPv6Col = $colSettings | Where-Object {($_.Caption -eq "ManualIPv6")} | Select-Object $colProperty

    foreach ($Setting in $colSettings)
    {
        # Get the DHCP enable value for the specific adapter
        $EnableDHCP = $EnableDHCPCol | Where-Object {($_.Name -eq $Setting.Name)}

        # Get the DHCPv6 enable value, and the ManualIPv6 value
        # EnableDHCPv6 is 0 (disabled) only if the "Managed Address Configuration Flag" and the "Other Stateful Configuration Flag" are both 0
        # (see https://blogs.technet.microsoft.com/teamdhcp/2009/03/03/dhcpv6-understanding-of-address-configuration-in-automatic-mode-and-installation-of-dhcpv6-server/)
        # ManualIPv6 is 1 (manual) only if the PrefixOrigin and SuffixOrigin of any IPv6 address are both "Manual" - other values indicate autoconfiguration
        $EnableDHCPv6 = $EnableDHCPv6Col | Where-Object {($_.Name -eq $Setting.Name)}
        $ManualIPv6 = $ManualIPv6Col | Where-Object {($_.Name -eq $Setting.Name)}

        $bSaveIPv6Settings = $false
        # if EnableDHCPv6 = 0 and ManualIPv6 = 1, we want to save IPv6 addresses and settings. Otherwise, all IPv6 addresses were autoconfigured and we don't need to save/restore them
        if ($EnableDHCPv6.CurrentValue -eq 0 -and $ManualIPv6.CurrentValue -eq 1)
        {
            $bSaveIPv6Settings = $true
        }

        # Save the IP Settings in the IP config file
        $bIPSetting = SaveIPSetting $Setting "Vlan" $EnableDHCP.CurrentValue $bSaveIPv6Settings

        if ($bIPSetting -eq $false)
        {
            WriteObjectToConfigFile $Setting $colProperty
        }
    }

    WriteTextToConfigFile "VlanSettingsEnd"
    WriteTextToConfigFile ""

    WriteTextToIPFile "VlanIPSettingsEnd"
    WriteTextToIPFile ""
}

#Save NICPART settings
Function SaveNICPARTSettings
{
    Write-Log $MyInvocation.MyCommand
    $colProperty = "Name","PartitionNumber","Identifier","MinBWPercent","MaxBWPercent"

    WriteTextToConfigFile "NICPARTSettingsStart"

    #start lock
    BeginApply

    try
    {
        #Get the partition information for all partitions on system and save them
        $PartitionArray = InvokeMethod $Script:service GetPartitionsForPort "szDeviceID" $null
            WriteObjectToConfigFile $PartitionArray.Partitions $colProperty
    }
    catch
    {
        Write-log "Unable to Save NIC Partition Settings"
    }

    #release lock
    ReleaseLock

    WriteTextToConfigFile "NICPARTSettingsEnd"
    WriteTextToConfigFile ""
}

#Remove teams and vlans on system
Function RemoveTeamsAndVlans
{
    Write-Log $MyInvocation.MyCommand

    $IANetObjs = @('IANet_TeamOfAdapters','IANet_Vlan')

    foreach ($IANetObj in $IANetObjs)
    {
        $colDevices = @()
        $colDevices += GetObject $IANetObj

        BeginApply

        foreach ($Device in $colDevices)
        {
            try
            {
                DeleteObject $Device
            }
            catch
            {
                Write-Log "Unable to DeleteObject `$Device $Device"
            }
        }

        ReleaseLock
    }
}

# Check that adapters in configuration file are present and enabled on the system.
Function CheckAdaptersExist
{
    Write-Log $MyInvocation.MyCommand
    $Script:colImportAdapters  = ReadFromConfigFile "Adapters"
    if ($null -ne $Script:colImportAdapters )
    {
        foreach ($Adapter in $Script:colImportAdapters)
        {
            # Find the matching file adapter using the appropriate property to compare name,address or ID
            if ($Script:BDF -eq $false)
            {
                if ($bUpRestore -eq $true)
                {
                    $ImportedAdapter = $Script:EnabledSystemAdapters | Where-Object {($_.PermanentAddress -eq $Adapter.PermanentAddress)}
                }
                else
                {
                    $ImportedAdapter = $Script:EnabledSystemAdapters | Where-Object {($_.PCIDeviceID -eq $Adapter.PCIDeviceID)}
                }

                if ($null -eq $ImportedAdapter)
                {
                    $ImportedAdapter = $Script:EnabledSystemAdapters | Where-Object {($_.Name -eq $Adapter.Name)}
                }
            }
            else
            {
                $ImportedAdapter = $Script:EnabledSystemAdapters | Where-Object {($_.SlotID -eq $Adapter.SlotID)}
            }


            if ($null -eq $ImportedAdapter)
            {
                Write-Warning UnableToFindDevice
                Write-Log "Invalid `$Adapter = $Adapter"
            }
        }
    }
}

# Restore custome OEM value
Function RestoreOEMCustomizeableSetting
{
    Write-Log $MyInvocation.MyCommand
    $ImportOEMSetting = ReadFromConfigFile "OEMSetting"
    if ($null -ne $ImportOEMSetting)
    {
        Write-Host $Messages.SettingOEMValue
        Set-ItemProperty -Path HKLM:\SOFTWARE\Intel\Network_Services\NCS2 -Name OEMCustomizeable -Value $ImportOEMSetting.OEMCustomizeable -Type DWord
    }
}

Function RestoreSettings($Setting, $stringClass, $enumClass)
{
    if ($Setting)
    {
        # IPv6 Address and DefaultGateway must be set with netsh, currently no API to set setting in Middleware
        if ($Setting.Caption -eq "IPv6Address")
        {
            # netsh interface ipv6 add address $ConnectionName $Address
            $ConnectionName = Get-WmiObject $stringClass -Namespace "root/intelncs2" | Where-Object {($_.Name -eq $Setting.Name) -and ($_.Caption -eq "ConnectionName")} | Select-Object -ExpandProperty CurrentValue
            $ConnectionInterfaceIndex = Get-WmiObject Win32_NetworkAdapter -Namespace "root/cimv2" | Where-Object {($_.NetConnectionID -eq $ConnectionName)} | Select-Object -ExpandProperty InterfaceIndex


            # delete current IPv6 addresses to overwrite them with saved config, if there are no addresses then we catch the exception and continue since there is nothing to delete.
            try
            {
                $AllAddresses = Get-WmiObject Win32_NetworkAdapterConfiguration -Namespace "root/cimv2" | Where-Object {($_.InterfaceIndex -eq $ConnectionInterfaceIndex)} | Select-Object -ExpandProperty IPAddress -ErrorAction SilentlyContinue
            }
            catch
            {
                Write-Log "No IPv6Address found."
            }

            $IPv6Array = @()
            foreach ($Address in $AllAddresses)
            {
                if (-not $Address.Contains("."))
                {
                    $IPv6Array += $Address
                }
            }

            #wrap interfaceindex in quotes for netsh
            $ConnectionInterfaceIndex = "`"$($ConnectionInterfaceIndex)`""

            foreach ($IP in $IPv6Array)
            {
                $DeleteCommand = "netsh interface ipv6 delete address $ConnectionInterfaceIndex $IP"
                Invoke-Expression $DeleteCommand
            }

            foreach ($Value in $Setting.CurrentValues) #CurrentValues is a list of IPv6Addresses split by comma, set each address
            {
                $Command = "netsh interface ipv6 add address $ConnectionInterfaceIndex $Value"
                Invoke-Expression $Command
            }
        }

        elseif ($Setting.Caption -eq "DefaultGatewayIPv6")
        {
            # netsh interface ipv6 add route ::/0 $ConnectionName $Address
            $ConnectionName = Get-WmiObject $stringClass -Namespace "root/intelncs2" | Where-Object {($_.Name -eq $Setting.Name) -and ($_.Caption -eq "ConnectionName")} | Select-Object -ExpandProperty CurrentValue
            $ConnectionInterfaceIndex = Get-WmiObject Win32_NetworkAdapter -Namespace "root/cimv2" | Where-Object {($_.NetConnectionID -eq $ConnectionName)} | Select-Object -ExpandProperty InterfaceIndex

            # delete current IPv6 gateways to overwrite them with saved config, if there are no gateways then we catch the exception and continue since there is nothing to delete.
            try
            {
                $AllGateways = Get-WmiObject Win32_NetworkAdapterConfiguration -Namespace "root/cimv2" | Where-Object {($_.InterfaceIndex -eq $ConnectionInterfaceIndex)} | Select-Object -ExpandProperty DefaultIPGateway -ErrorAction SilentlyContinue
            }
            catch
            {
                Write-Log "No DefaultGatewayIPv6 found."
            }

            $IPv6Array = @()
            foreach ($Address in $AllGateways)
            {
                if (-not $Address.Contains("."))
                {
                    $IPv6Array += $Address
                }
            }

            #wrap interfaceindex in quotes for netsh
            $ConnectionInterfaceIndex = "`"$($ConnectionInterfaceIndex)`""

            foreach ($IP in $IPv6Array)
            {
                $DeleteCommand = "netsh interface ipv6 delete route ::/0 $ConnectionInterfaceIndex $IP"
                Invoke-Expression $DeleteCommand
            }

            foreach ($Value in $Setting.CurrentValues)
            {
                $Command = "netsh interface ipv6 add route ::/0 $ConnectionInterfaceIndex $Value"
                Invoke-Expression $Command
            }
        }

        elseif ($Setting.Caption -eq "EnableDHCPv6")
        {
            $EnableDHCPv6Setting = Get-WmiObject $enumClass -Namespace "root/intelncs2" | Where-Object {($_.Name -eq $Setting.Name) -and ($_.Caption -eq "EnableDHCPv6")} | Select-Object -ExpandProperty CurrentValue

            if (($Setting.CurrentValue -eq "0") -and ($EnableDHCPv6Setting.CurrentValue -eq "1"))
            {
                $ConnectionName = Get-WmiObject $stringClass -Namespace "root/intelncs2" | Where-Object {($_.Name -eq $Setting.Name) -and ($_.Caption -eq "ConnectionName")} | Select-Object -ExpandProperty CurrentValue
                $ConnectionInterfaceIndex = Get-WmiObject Win32_NetworkAdapter -Namespace "root/cimv2" | Where-Object {($_.NetConnectionID -eq $ConnectionName)} | Select-Object -ExpandProperty InterfaceIndex

                $ConnectionInterfaceIndex = "`"$($ConnectionInterfaceIndex)`""

                $dhcpv6routercmd = "netsh int ipv6 set int $ConnectionInterfaceIndex routerdiscovery=disabled managedaddress=disabled otherstateful=disabled"

                Invoke-Expression $dhcpv6routercmd
                Start-Sleep -s 1
            }

            elseif (($Setting.CurrentValue -eq "1") -and ($EnableDHCPv6Setting.CurrentValue -eq "0"))
            {
                $ConnectionName = Get-WmiObject $stringClass -Namespace "root/intelncs2" | Where-Object {($_.Name -eq $Setting.Name) -and ($_.Caption -eq "ConnectionName")} | Select-Object -ExpandProperty CurrentValue
                $ConnectionInterfaceIndex = Get-WmiObject Win32_NetworkAdapter -Namespace "root/cimv2" | Where-Object {($_.NetConnectionID -eq $ConnectionName)} | Select-Object -ExpandProperty InterfaceIndex

                $ConnectionInterfaceIndex = "`"$($ConnectionInterfaceIndex)`""

                $dhcpv6managedcmd = "netsh int ipv6 set int $ConnectionInterfaceIndex managedaddress=enabled"

                Invoke-Expression $dhcpv6managedcmd
                Start-Sleep -s 1
            }
        }
        else
        {
            $result = SetSetting $Setting
            Write-Log "SetSetting $result"
        }
    }
}

# Restore adapter settings
Function RestoreAdapterSettings($colImportedSettings)
{
    Write-Log $($MyInvocation.MyCommand)
    Write-Log "`$colImportedSettings = $colImportedSettings"
    # Get adapter settings on system
    $colSetting = GetObject "IANet_AdapterSetting" | Sort-Object -Property Name,Caption
    # Add parent identifiers to settings on system
    foreach ($cSetting in $colSetting)
    {
        AddParentIDs $cSetting
    }

    # start lock
    BeginApply

    # Loop through the passed in settings from the config file
    foreach ($ImportedSetting in $colImportedSettings) #for each setting in AdapterIPSettings
    {
        try
        {
            # Find the matching system setting using the appropriate property to compare name,address or ID
            if ($Script:BDF -eq $false)
            {
                # finds the setting with the same caption and device name
                if (($bUpRestore -eq $true))
                {
                    $Setting = $colSetting | Where-Object {($_.PermanentAddress -eq $ImportedSetting.PermanentAddress) -and ($_.Caption -eq $ImportedSetting.Caption)}
                }
                else
                {
                    $Setting = $colSetting | Where-Object {($_.PCIDeviceID -eq $ImportedSetting.PCIDeviceID) -and ($_.Caption -eq $ImportedSetting.Caption)}
                }

                if ($null -eq $Setting)
                {
                    $Setting = $colSetting | Where-Object {($_.Name -eq $ImportedSetting.Name) -and ($_.Caption -eq $ImportedSetting.Caption)}
                }
            }
            else
            {
                $Setting = $colSetting | Where-Object {($_.SlotID -eq $ImportedSetting.SlotID) -and ($_.Caption -eq $ImportedSetting.Caption)}
            }

            # If the setting in the file can't be found on the system, continue to the next setting
            if ($null -eq $Setting)
            {
                Write-Log "Not Restored: $ImportedSetting"
                continue
            }

            # See if 'CurrentValues' is a property over the system setting, if so set that value from the config file setting
            if (($Setting.Properties | Select-Object -ExpandProperty Name) -contains "CurrentValues")
            {
                # If no value was present from the file setting, then set it to empty
                if ($null -eq $ImportedSetting.CurrentValues)
                {
                    $Setting.CurrentValues = {}
                }
                else
                {
                    $Setting.CurrentValues = $ImportedSetting.CurrentValues.Split(",")
                }
            }
            else
            {
                $Setting.CurrentValue = $ImportedSetting.CurrentValue
            }

            RestoreSettings $Setting "IANet_AdapterSettingString" "IANet_AdapterSettingEnum"
        }
        catch
        {
            $ErrorMessagePart1 = [string]$Messages.UnableToSetPart1
            $ErrorMessagePart2 = [string]$Messages.UnableToSetPart2
            # Unable to set ... on ...
            Write-Log "$ErrorMessagePart1 $($Setting.Caption) $ErrorMessagePart2 $($Setting.Name)"
        }
    }

    ReleaseLock
}

# Set the primary and secondary adapters on the team
Function RestorePrimaryAndSecondary($Team)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log "`$Team = $Team"
    Write-Host $Messages.RestoringTeamAdapters $Team.TeamName

    # Find the adapter objects from the configuration file that match the team primary and secondary adapter names
    $PrimaryImportAdapter = $Script:colImportAdapters | Where-Object {$_.OriginalDisplayName -eq $Team.PrimaryAdapter}
    $SecondaryImportAdapter = $Script:colImportAdapters | Where-Object {$_.OriginalDisplayName -eq $Team.SecondaryAdapter}

    # Find the matching system adapter using the appropriate property to compare name,address or ID
    if ($Script:BDF -eq $false)
    {
        if ($bUpRestore)
        {
            $PrimaryAdapter = $Script:EnabledSystemAdapters | Where-Object {($_.PermanentAddress -eq $PrimaryImportAdapter.PermanentAddress)}
        }
        else
        {
            $PrimaryAdapter = $Script:EnabledSystemAdapters | Where-Object {($_.PCIDeviceID -eq $PrimaryImportAdapter.PCIDeviceID)}
        }
    }
    else
    {
        $PrimaryAdapter = $Script:EnabledSystemAdapters | Where-Object {($_.SlotID -eq $PrimaryImportAdapter.SlotID)}
    }

    if ($Script:BDF -eq $false)
    {
        if ($bUpRestore -eq $true)
        {
            $SecondaryAdapter = $Script:EnabledSystemAdapters | Where-Object {($_.PermanentAddress -eq $SecondaryImportAdapter.PermanentAddress)}
        }
        else
        {
            $SecondaryAdapter = $Script:EnabledSystemAdapters | Where-Object {($_.PCIDeviceID -eq $SecondaryImportAdapter.PCIDeviceID)}
        }
    }
    else
    {
        $SecondaryAdapter = $Script:EnabledSystemAdapters | Where-Object {($_.SlotID -eq $SecondaryImportAdapter.SlotID)}
    }

    # Set the primary and secondary adapters on the team
    $Team = GetObject "IANet_TeamOfAdapters" | Where-Object {$_.TeamName -eq $Team.TeamName} | Sort-Object -Property TeamName
    $param = $Team.psbase.GetMethodParameters("ValidateAndSetAdapterPriority")
    $param.PrimaryAdapter = $PrimaryAdapter
    $param.SecondaryAdapter = $SecondaryAdapter
    $param.SetValuesOnNoError = 2
    $value = $Team.psbase.InvokeMethod("ValidateAndSetAdapterPriority", $param, $invokeoptions)
    Write-Log $Value
}

# Restore teams from configuration file
Function RestoreTeams
{
    Write-Log $MyInvocation.MyCommand
    # Get the team objects from the configuration file
    $colImportTeams = ReadFromConfigFile "Teams"

    if ($null -ne $colImportTeams)
    {
        BeginApply

        foreach ($Team in $colImportTeams)
        {
            Write-Host $Messages.Restoring $Team.TeamName

            try
            {
                # Create a new team object
                $classdef = NewObject root\intelncs2:IANet_TeamOfAdapters

                # fill in the CreateTeam's method parameters with the new object
                $param = $classdef.psbase.GetMethodParameters("CreateTeam2Ex")

                $TeamedAdapters = $Team.TeamMembers.split("|")
                $param.Adapters = @()

                $bCreateTeam = $true
                # Set the teammembers for creation
                foreach ($Adapter in $TeamedAdapters)
                {
                    $Adapter = $Adapter.Split(",")
                    $AdapterName = $Adapter[0]
                    $AdapterPermanentAddress = $Adapter[1]
                    $AdapterPCIDeviceID = $Adapter[2]
                    $AdapterSlotID = $Adapter[3]
                    $Adapter = $null

                    # Find the matching system adapter using the appropriate property to compare address or ID
                    if ($Script:BDF -eq $false)
                    {
                        if ($bUpRestore -eq $true)
                        {
                            $Adapter = $Script:EnabledSystemAdapters | Where-Object {($_.PermanentAddress -eq $AdapterPermanentAddress)}
                        }
                        else
                        {
                            $Adapter = $Script:EnabledSystemAdapters | Where-Object {($_.PCIDeviceID -eq $AdapterPCIDeviceID)}
                        }

                        # Older versions of SaveRestore.ps1 won't have the Adapter's PermanentAddress, PCIDeviceID, or SlotID saved, so use the name.
                        if ($null -eq $Adapter)
                        {
                            $Adapter = $Script:EnabledSystemAdapters | Where-Object {($_.Name -eq $AdapterName)}
                        }
                    }
                    else
                    {
                        $Adapter = $Script:EnabledSystemAdapters | Where-Object {($_.SlotID -eq $AdapterSlotID)}
                    }

                    # If a teammember can't be found output an error and breakout of team creation
                    if ($null -eq $Adapter)
                    {
                        Write-Warning UnableToFindDevice
                        Write-Log $AdapterName
                        $bCreateTeam = $false
                        break
                    }
                    $param.Adapters += $Adapter
                }

                # If an error was found, don't try and create the team
                if ($bCreateTeam -eq $false)
                {
                    Continue
                }

                $param.TeamMode = $Team.TeamMode
                $param.TeamName = $Team.TeamName

                # the invoke options
                $invokeoptions = New-Object System.Management.InvokeMethodOptions
                $invokeoptions.Context = $Script:context

                # call the CreateTeam method to create the new team
                $null = $classdef.psbase.InvokeMethod("CreateTeam2Ex", $param, $invokeoptions)

                # Set primary and secondary adapters
                RestorePrimaryAndSecondary $Team
            }
            catch
            {
                Write-Log "Unable to Restore Teams"
            }
        }

        ReleaseLock
    }
}

#Restore team settings
Function RestoreTeamSettings($colImportSettings)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log "`$colImportSettings = $colImportSettings"
    # Get the current team settings
    $colSetting = GetObject "IANet_TeamSetting" | Sort-Object -Property Name,Caption

    BeginApply

    foreach ($Setting in $colSetting)
    {
        try
        {
            # If the current setting using 'CurrentValues' then set that value.
            if (($Setting.Properties | Select-Object -ExpandProperty Name) -contains "CurrentValues")
            {
                $ValueObject = $colImportSettings | Where-Object {($_.TeamName -eq $Setting.Name) -and ($_.Caption -eq $Setting.Caption)} | Select-Object CurrentValues
                if ($null -eq $ValueObject.CurrentValues)
                {
                    $Setting.CurrentValues = {}
                }
                else
                {
                    $Setting.CurrentValues = $ValueObject.CurrentValues.Split(",")
                }
            }
            else
            {
                $ValueObject = $colImportSettings | Where-Object {($_.TeamName -eq $Setting.Name) -and ($_.Caption -eq $Setting.Caption)} | Select-Object CurrentValue
                $Setting.CurrentValue = $ValueObject.CurrentValue
            }

            #If a matching system setting was found set it to the restore value
            if($null -ne $ValueObject)
            {
                RestoreSettings $Setting "IANet_TeamSettingString" "IANet_TeamSettingEnum"
            }

        }
        catch
        {
            $ErrorMessagePart1 = [string]$Messages.UnableToSetPart1
            $ErrorMessagePart2 = [string]$Messages.UnableToSetPart2
            # Unable to set ... on ...
            Write-Log "$ErrorMessagePart1 $($Setting.Caption) $ErrorMessagePart2 $($Setting.Name)"
        }
    }

    ReleaseLock
}

Function RestoreVlans
{
    Write-Log $MyInvocation.MyCommand
    # Get vlan objects from config file
    $colImportVlans = ReadFromConfigFile "Vlans"

    if ($null -ne $colImportVlans)
    {
        foreach ($Vlan in $colImportVlans)
        {
            # Start lock
            BeginApply
            try
            {
                # Find the matching system adapter using the appropriate property to compare name,address or ID
                if ($Script:BDF -eq $false)
                {
                    if ($bUpRestore -eq $true)
                    {
                        $Adapter = $Script:Adapters | Where-Object {($_.PermanentAddress -eq $Vlan.ParentPermanentAddress)}
                    }
                    else
                    {
                        $Adapter = $Script:Adapters | Where-Object {($_.PCIDeviceID -eq $Vlan.ParentPCIDeviceID)}
                    }

                    # Older versions of SaveRestore.ps1 won't have the Adapter's PermanentAddress, PCIDeviceID, or SlotID saved, so use the name.
                    if ($null -eq $Adapter)
                    {
                        $Adapter = $Script:Adapters | Where-Object {($_.Name -eq $Vlan.ParentName)}
                    }
                }
                else
                {
                    $Adapter = $Script:Adapters | Where-Object {($_.SlotID -eq $Vlan.ParentSlotID)}
                }

                if ($null -ne $Adapter)
                {
                    # Get the vlan creation class associated with adapter parent object
                    $VlanCreator = GetAssociated $Adapter.path.path "ResultClass = IANet_802dot1QVLANService" | Where-Object {1}
                }
                # Check if vlan needs to be created on a team
                else
                {
                    # Get the logical ethernet adapter object for the team parent
                    $Team = GetObject "IANet_LogicalEthernetAdapter" | Where-Object {( NScompare $_.Name $Vlan.ParentName )}

                    if ($null -ne $Team)
                    {
                        # Get the vlan creation class associated with team parent object
                        $VlanCreator = GetAssociated $Team.path.path "ResultClass = IANet_802dot1QVLANService" | Where-Object {1}
                    }
                }

                # If the vlan creation class was found continue on to create the vlan
                if ($null -ne $VlanCreator)
                {
                    Write-Host $Messages.Restoring $Vlan.VlanName

                    # fill in the CreateVlan's method parameters
                    $param = $VlanCreator.psbase.GetMethodParameters("CreateVlan")

                    $param.Name = $Vlan.VLANNAME
                    $param.VLANNumber = $Vlan.VLANID

                    # the invoke options
                    $invokeoptions = New-Object System.Management.InvokeMethodOptions
                    $invokeoptions.Context = $Script:context

                    # call the CreateVlan method to create the new vlan
                    $null = $VlanCreator.psbase.InvokeMethod("CreateVlan", $param, $invokeoptions)
                }
            }
            catch
            {
                Write-Log "Unable to Restore VLANs"
            }
            ReleaseLock
        }
    }
}

# Restore vlan settings
Function RestoreVlanSettings($colImportSettings)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log "`$colImportSettings = $colImportSettings"
    # Get the current Vlan settings
    $vlanSettings = GetObject "IANet_VLANSetting" | Sort-Object -Property ParentName,VLANID,Caption

    # Start lock
    BeginApply

    foreach ($Setting in $vlanSettings)
    {
        try
        {
            if ($Script:BDF -eq $false)
            {
                #finds the setting with the same caption and device name
                if ($bUpRestore -eq $true)
                {
                    $ValueObject = $colImportSettings | Where-Object {($_.ParentPermanentAddress -eq $Setting.ParentPermanentAddress) -and ($_.Caption -eq $Setting.Caption) -and ($_.VLANID -eq $Setting.VLANID)}
                }
                else
                {
                    $ValueObject = $colImportSettings | Where-Object {($_.ParentPCIDeviceID -eq $Setting.ParentPCIDeviceID) -and ($_.Caption -eq $Setting.Caption) -and ($_.VLANID -eq $Setting.VLANID)}
                }

                # Older versions of SaveRestore.ps1 won't have the Adapter's PermanentAddress, PCIDeviceID, or SlotID saved, so use the name.
                if ($null -eq $ValueObject)
                {
                    $ValueObject = $colImportSettings | Where-Object {( NSCompare $_.ParentName $Setting.ParentName ) -and ( $_.Caption -eq $Setting.Caption) -and ($_.VLANID -eq $Setting.VLANID)}
                }
            }
            else
            {
                $ValueObject = $colImportSettings | Where-Object {($_.ParentSlotID -eq $Setting.ParentSlotID) -and ($_.Caption -eq $Setting.Caption) -and ($_.VLANID -eq $Setting.VLANID)}
            }

            if (($Setting.Properties | Select-Object -ExpandProperty Name) -contains "CurrentValues")
            {
                if ($null -eq $ValueObject.CurrentValues)
                {
                    $Setting.CurrentValues = {}
                }
                else
                {
                    $Setting.CurrentValues = $ValueObject.CurrentValues.Split(",")
                }
            }
            else
            {
                $Setting.CurrentValue = $ValueObject.CurrentValue
            }

            # If a matching system setting was found set it to the restore value
            if($null -ne $ValueObject)
            {
                RestoreSettings $Setting "IANet_VLANSettingString" "IANet_VLANSettingEnum"
            }

        }
        catch
        {
            $ErrorMessagePart1 = [string]$Messages.UnableToSetPart1
            $ErrorMessagePart2 = [string]$Messages.UnableToSetPart2
            # Unable to set ... on ...
            Write-Log "$ErrorMessagePart1 $($Setting.Caption) $ErrorMessagePart2 $($Setting.Name)"
        }
    }

    ReleaseLock
}

# Restore the IP settings from the IP config file
Function RestoreIPSettings
{
    Write-Log $MyInvocation.MyCommand
    # Restore Adapter IP settings from the IP config file
    $colImportSettings = ReadFromIPFile "AdapterIPSettings"

    if ($null -ne $colImportSettings)
    {
        foreach ($setting in $colImportSettings)
        {
            # If the staticip is set on a disconnected adapter from cmdline tools (netsh, New-NetIPAddress, Set-NetIPInterface, etc)
            # the private 169.254.x.x IP that Microsoft assigns when it can't get an address from the DHCP server
            # comes along and saverestore ends up restoring that IP with a 255.255.0.0 subnetmask instead.
            # Only capture the first ip, which is what static ip and subnetmask the user intended to set
            # ignore the 169.254.x.x IP and 255.255.0.0 subnetmask
            if (($setting.Caption -eq 'IPv4Address') -or
                ($setting.Caption -eq 'SubnetMask'))
            {
                if (($setting.CurrentValues.ToCharArray()) -contains [char]',')
                {
                    $setting.CurrentValues = $setting.CurrentValues.Substring(0, $setting.CurrentValues.lastIndexOf(','))
                }
            }
        }

        Write-Host $Messages.RestoringAdapterIP
        RestoreAdapterSettings $colImportSettings
    }

    if ($Script:bANSInstalled)
    {
        # Restore Team IP settings from the IP config file
        $colImportSettings = ReadFromIPFile "TeamIPSettings"
        if ($null -ne $colImportSettings)
        {
            Write-Host $Messages.RestoringTeamIP
            RestoreTeamSettings $colImportSettings $colSetting
        }

        # Restore Vlan IP settings from the IP config file
        $colImportSettings = ReadFromIPFile "VlanIPSettings"
        if ($null -ne $colImportSettings)
        {
            Write-Host $Messages.RestoringVLANIP
            RestoreVlanSettings $colImportSettings
        }
    }
}

# Restore devices and settings from configuration files
Function RestoreAdvancedSettingsAndDevices
{
    Write-Log $MyInvocation.MyCommand
    #Restore Adapter avanced settings from config file
    $colImportSettings = ReadFromConfigFile "AdapterSettings"
    if ($null -ne $colImportSettings)
    {
        Write-Host $Messages.RestoringAdapterSettings
        RestoreAdapterSettings $colImportSettings
    }

    if (!$Script:bANSInstalled)
    {
        Write-Warning UnableToFindANS
    }
    else
    {
        #Restore Teams
        RestoreTeams

        #Restore Team advanced settings from config file
        $colImportSettings = ReadFromConfigFile "TeamSettings"
        if ($null -ne $colImportSettings)
        {
            Write-Host $Messages.RestoringTeamSettings
            RestoreTeamSettings $colImportSettings
        }

        #Restore Vlans
        RestoreVlans

        #Restore Vlan avanced settings from config file
        $colImportSettings = ReadFromConfigFile "VlanSettings"
        if ($null -ne $colImportSettings)
        {
            Write-Host $Messages.RestoringVLANSettings
            RestoreVlanSettings $colImportSettings
        }
    }

    if (Test-Path $Script:ScriptPath\Saved_StaticIP.txt)
    {
        #Restore IP Settings from IP config file
        RestoreIPSettings
    }
}

#Restore NICPART settings
Function RestoreNICPARTSettings
{
    Write-Log $MyInvocation.MyCommand
    #Restore NICPART settings from config file
    $colImportPartitions = ReadFromConfigFile "NICPARTSettings"
    Write-Host $Messages.RestoringAdapterPartition

    BeginApply

    try
    {
        if ($null -ne $colImportPartitions)
        {
            #Get the partition information for all partitions on system and save them
            $PartitionObject = InvokeMethod $Script:service GetPartitionsForPort "szDeviceID" $null
            $PartitionArray = $PartitionObject.Partitions


            foreach ($Partition in $PartitionArray)
            {
                $ImportPartition = $colImportPartitions | Where-Object {($_.PartitionNumber -eq $Partition.PartitionNumber) -and ($_.Identifier -eq $Partition.Identifier)}

                if ($null -ne $ImportPartition)
                {
                    if ($Script:BDF -eq $true)
                    {
                        $Partition.Name = "IGNORE"
                        $Partition.DeviceGuid = "IGNORE"
                    }
                    else
                    {
                        $Partition.Identifier = "IGNORE"
                        $Partition.DeviceGuid = "IGNORE"
                    }

                    $Partition.MinBWPercent = $ImportPartition.MinBWPercent
                    $Partition.MaxBWPercent = $ImportPartition.MaxBWPercent
                }
            }

            $Output = InvokeMethod $Script:service ValidateAndSetBAndwidthsForPort "Partitions","SetValuesOnNoError" $PartitionArray,"1"
        }
    }
    catch
    {
        Write-Log "Unable to Restore NIC Partition Settings"
    }

    ReleaseLock
}


<#
.SYNOPSIS
    Make sure users know about BDF parameter when using NICPART enabled adapters
.DESCRIPTION
    By default, the save restore script uses the adapter's friendly name to match the name of the adapter
    in the configuration file to adapter in the system.  Because of the way Windows enumerates devices and
    assigns friendly names, the name of the adapter saved in the configuration file may not be the exact
    same adapter/port/partition on another system. The /bdf command line option should be used when
    restoring settings of NICPART enabled devices on multiple systems.
.NOTES
    This must be invoked after GetAdaptersOnSystem - as it relies on the contents of $Script:Adapters.
    The global variable is used to save time when restoring settings on user systems (to avoid polling
    the system again just to get the same data).
#>
function CheckNICPART_BDFWarning
{
    Write-Log $MyInvocation.MyCommand
    $NICPART = $Script:Adapters | Where-Object {($_.Capabilities -eq 76)}
    if ($NICPART -and $Script:BDF -eq $false)
    {
        Write-Host $Messages.NICPartitioningDetected
    }

}

<#
.SYNOPSIS
    "No Space Compare" Compares two strings without whitespace
.DESCRIPTION
    Intel(R) Network Connections Software recently introduced a change to ANS team prefixes. The old format
    was "TEAM : ". The new format is "TEAM: ". To preserve user configurations during an upgrade it is
    necessary to compare some devices without space differences.
#>
function NSCompare ($s1, $s2)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log "Comparing `$s1=$s1 and `$s2=$s2..."
    $s1temp = $s1 -replace '\s', ''
    $s2temp = $s2 -replace '\s', ''
    return ($s1temp -eq $s2temp)
}

#-----------------------------------------------------[Execution]------------------------------------------------------

# Determine which action was specified by the user
switch($Script:Action)
{
    "Save" {$Script:bSave=$true
            Write-Host $Messages.PerformingSave
            break}
    "UpSave" {$Script:bUpSave=$true
            Write-Host $Messages.PerformingUpgradeSave
            break}
    "Restore" {$Script:bRestore=$true
            Write-Host $Messages.PerformingRestore
            break}
    "UpRestore" {$Script:bUpRestore=$true
            Write-Host $Messages.PerformingUpgradeRestore
            break}
    "Remove" {$Script:bRemove=$true
            Write-Host $Messages.PerformingRemove
            break}
    "?" {PrintUsage}
    "help"{PrintUsage}
    default {PrintUsage}
}

CheckForAdminRights
CheckForDMiXInstall
CheckForANSInstall

# Setup the save/restore path
SetupSaveRestoreLocation

# If a save is the primary action
if (($Script:bSave -eq $true) -or ($Script:bUpSave -eq $true))
{
    GetAdaptersOnSystem

    RemoveOldFiles

    SaveAdapters

    SaveOEMCustomizeableSetting

    SaveAdapterSettings

    if ($Script:bANSInstalled)
    {
        SaveTeams

        SaveTeamSettings

        SaveVlans

        SaveVlanSettings
    }

    SaveNICPARTSettings
}
# If a restore is the primary action
elseif (($Script:bRestore -eq $true) -or ($Script:bUpRestore -eq $true))
{
    CheckIfConfigFileExists

    if ($Script:bANSInstalled)
    {
        RemoveTeamsAndVlans
    }

    GetAdaptersOnSystem

    CheckNICPART_BDFWarning

    CheckAdaptersExist

    RestoreOEMCustomizeableSetting

    RestoreAdvancedSettingsAndDevices

    RestoreNICPARTSettings
}
# If remove is the primary action
elseif ($Script:bRemove -eq $true)
{
    RemoveTeamsAndVlans
}


# SIG # Begin signature block
# MIIotgYJKoZIhvcNAQcCoIIopzCCKKMCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCAg5XcJXSwWFqph
# dupHD17uM3jd+S7JJX0NITYAtsMChqCCEgUwggWeMIIEhqADAgECAhEAzS1l4rws
# CIvYBjRVawV4ujANBgkqhkiG9w0BAQsFADB8MQswCQYDVQQGEwJHQjEbMBkGA1UE
# CBMSR3JlYXRlciBNYW5jaGVzdGVyMRAwDgYDVQQHEwdTYWxmb3JkMRgwFgYDVQQK
# Ew9TZWN0aWdvIExpbWl0ZWQxJDAiBgNVBAMTG1NlY3RpZ28gUlNBIENvZGUgU2ln
# bmluZyBDQTAeFw0yMTA0MDIwMDAwMDBaFw0yMzA0MDIyMzU5NTlaMIGEMQswCQYD
# VQQGEwJVUzETMBEGA1UECAwKQ2FsaWZvcm5pYTEUMBIGA1UEBwwLU2FudGEgQ2xh
# cmExGjAYBgNVBAoMEUludGVsIENvcnBvcmF0aW9uMRIwEAYDVQQLDAlTb2xhcktN
# Q1MxGjAYBgNVBAMMEUludGVsIENvcnBvcmF0aW9uMIIBojANBgkqhkiG9w0BAQEF
# AAOCAY8AMIIBigKCAYEA7CCN9iKpDHOrRceKhlXFP/tf6Lllw2H2fR9KVI4/fQIx
# MU1hXwnlHmAzMCY7IgcCFY4p3F5/MJGKaqYngwOo28Zo6Q1N6ukysA7PSavmF2RY
# WD6VFeya/2H0PoNeRFjHaRzSeynFFeJAFew9r7UReUwM/507sxZYPQuWWIdAEK7H
# Dqp2VlHmgZOXVGHhNO6GFOKpC/C01g6X3x6OquddRNMt5UrZzZzDo5MpJz9SBB2V
# jiqwZ80dvNR2W2xi90cIHh4BkXvB54UNkp4VTVu16T0k3cweo+C39U7GrCAr5Axz
# DETjBvhNtP1sf9SoRV7xY6g5wssfI7yYT9J0gsifn/Vy8MWH355TPoA+PVhbAu0m
# 9FMz4EWu55nnUurNML2jaUxsos21/7ELat12kWC0tq9fhkODjKO8X9PuiBHflZLk
# d3F4QcSMvuGocWGqE77VV3vn8jlvigm2TOV0CfGTQajGMX0jeTRZ19fzBNkt2X9d
# SSGolI/Kj1gSvCggpkUBAgMBAAGjggGQMIIBjDAfBgNVHSMEGDAWgBQO4TqoUzox
# 1Yq+wbutZxoDha00DjAdBgNVHQ4EFgQUshkNuM2SdwJnW4vFy8c4FtUTrbQwDgYD
# VR0PAQH/BAQDAgeAMAwGA1UdEwEB/wQCMAAwEwYDVR0lBAwwCgYIKwYBBQUHAwMw
# EQYJYIZIAYb4QgEBBAQDAgQQMEoGA1UdIARDMEEwNQYMKwYBBAGyMQECAQMCMCUw
# IwYIKwYBBQUHAgEWF2h0dHBzOi8vc2VjdGlnby5jb20vQ1BTMAgGBmeBDAEEATBD
# BgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29S
# U0FDb2RlU2lnbmluZ0NBLmNybDBzBggrBgEFBQcBAQRnMGUwPgYIKwYBBQUHMAKG
# Mmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1JTQUNvZGVTaWduaW5nQ0Eu
# Y3J0MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG
# 9w0BAQsFAAOCAQEAVadLNRW4f/pKMqrbn0BdOoQ8/1EJ87gvVfosei2bLwTEvpmv
# mn2n561H6AFedtIJ6L4FmXII4M4r20i+5LREbI6PpKDmOAf4xW7POxfCRvkTQAZO
# 3zoVxjMQBXo7cZVF1xHCdviXzD1usuIiCF8DLm6z4O/kyeFFNcn816yPQct91Pnk
# SBBVvL+Kwu8xvR+ZIQy632WUA4HnNpRdFnVSzUifEg2GrtsKZR8k+rm2o8K8yjJq
# 3SznwgJQCMVMh3CtRtUwE/c7o/6rvm53fTYJDd3aoPHVgH6S2WqS3+3mQG7A6hTD
# nrP/mYnS4PF7XzxxjZhUlhy4G/MarJPvT9IrNDCCBfUwggPdoAMCAQICEB2iSDBv
# myYY0ILgln0z02owDQYJKoZIhvcNAQEMBQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpOZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkgQ2l0eTEeMBwGA1UEChMV
# VGhlIFVTRVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVVU0VSVHJ1c3QgUlNBIENl
# cnRpZmljYXRpb24gQXV0aG9yaXR5MB4XDTE4MTEwMjAwMDAwMFoXDTMwMTIzMTIz
# NTk1OVowfDELMAkGA1UEBhMCR0IxGzAZBgNVBAgTEkdyZWF0ZXIgTWFuY2hlc3Rl
# cjEQMA4GA1UEBxMHU2FsZm9yZDEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSQw
# IgYDVQQDExtTZWN0aWdvIFJTQSBDb2RlIFNpZ25pbmcgQ0EwggEiMA0GCSqGSIb3
# DQEBAQUAA4IBDwAwggEKAoIBAQCGIo0yhXoYn0nwli9jCB4t3HyfFM/jJrYlZilA
# hlRGdDFixRDtsocnppnLlTDAVvWkdcapDlBipVGREGrgS2Ku/fD4GKyn/+4uMyD6
# DBmJqGx7rQDDYaHcaWVtH24nlteXUYam9CflfGqLlR5bYNV+1xaSnAAvaPeX7Wpy
# vjg7Y96Pv25MQV0SIAhZ6DnNj9LWzwa0VwW2TqE+V2sfmLzEYtYbC43HZhtKn52B
# xHJAteJf7wtF/6POF6YtVbC3sLxUap28jVZTxvC6eVBJLPcDuf4vZTXyIuosB69G
# 2flGHNyMfHEo8/6nxhTdVZFuihEN3wYklX0Pp6F8OtqGNWHTAgMBAAGjggFkMIIB
# YDAfBgNVHSMEGDAWgBRTeb9aqitKz1SA4dibwJ3ysgNmyzAdBgNVHQ4EFgQUDuE6
# qFM6MdWKvsG7rWcaA4WtNA4wDgYDVR0PAQH/BAQDAgGGMBIGA1UdEwEB/wQIMAYB
# Af8CAQAwHQYDVR0lBBYwFAYIKwYBBQUHAwMGCCsGAQUFBwMIMBEGA1UdIAQKMAgw
# BgYEVR0gADBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8vY3JsLnVzZXJ0cnVzdC5j
# b20vVVNFUlRydXN0UlNBQ2VydGlmaWNhdGlvbkF1dGhvcml0eS5jcmwwdgYIKwYB
# BQUHAQEEajBoMD8GCCsGAQUFBzAChjNodHRwOi8vY3J0LnVzZXJ0cnVzdC5jb20v
# VVNFUlRydXN0UlNBQWRkVHJ1c3RDQS5jcnQwJQYIKwYBBQUHMAGGGWh0dHA6Ly9v
# Y3NwLnVzZXJ0cnVzdC5jb20wDQYJKoZIhvcNAQEMBQADggIBAE1jUO1HNEphpNve
# aiqMm/EAAB4dYns61zLC9rPgY7P7YQCImhttEAcET7646ol4IusPRuzzRl5ARokS
# 9At3WpwqQTr81vTr5/cVlTPDoYMot94v5JT3hTODLUpASL+awk9KsY8k9LOBN9O3
# ZLCmI2pZaFJCX/8E6+F0ZXkI9amT3mtxQJmWunjxucjiwwgWsatjWsgVgG10Xkp1
# fqW4w2y1z99KeYdcx0BNYzX2MNPPtQoOCwR/oEuuu6Ol0IQAkz5TXTSlADVpbL6f
# ICUQDRn7UJBhvjmPeo5N9p8OHv4HURJmgyYZSJXOSsnBf/M6BZv5b9+If8AjntIe
# Q3pFMcGcTanwWbJZGehqjSkEAnd8S0vNcL46slVaeD68u28DECV3FTSK+TbMQ5Lk
# uk/xYpMoJVcp+1EZx6ElQGqEV8aynbG8HArafGd+fS7pKEwYfsR7MUFxmksp7As9
# V1DSyt39ngVR5UR43QHesXWYDVQk/fBO4+L4g71yuss9Ou7wXheSaG3IYfmm8SoK
# C6W59J7umDIFhZ7r+YMp08Ysfb06dy6LN0KgaoLtO0qqlBCk4Q34F8W2WnkzGJLj
# tXX4oemOCiUe5B7xn1qHI/+fpFGe+zmAEc3btcSnqIBv5VPU4OOiwtJbGvoyJi1q
# V3AcPKRYLqPzW0sH3DJZ84enGm1YMIIGZjCCBE6gAwIBAgITMwAAAES3P/zvWs+i
# egAAAAAARDANBgkqhkiG9w0BAQUFADB/MQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMSkwJwYDVQQDEyBNaWNyb3NvZnQgQ29kZSBWZXJpZmljYXRp
# b24gUm9vdDAeFw0xNTA3MjIyMTAzNDlaFw0yNTA3MjIyMTAzNDlaMIGIMQswCQYD
# VQQGEwJVUzETMBEGA1UECBMKTmV3IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5IENp
# dHkxHjAcBgNVBAoTFVRoZSBVU0VSVFJVU1QgTmV0d29yazEuMCwGA1UEAxMlVVNF
# UlRydXN0IFJTQSBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eTCCAiIwDQYJKoZIhvcN
# AQEBBQADggIPADCCAgoCggIBAIASZRc2DsPbCLPQrFcNdu3NJ9NMrVCDYeKqIE0J
# LWQJ3M6Jn8w9qez2z8Hc8dOx1ns3KBErR9o5xrw6GbRfpr19naNjQrZ28qk7K5H4
# 4m/Q7BYgkAk+4uh0yRi0kdRiZNt/owbxiBhqkCI8vP4T8IcUe/bkH47U5FHGEWdG
# CFHLhhRUP7wz/n5snP8WnRi9UY41pqdmyHJn2yFmsdSbeAPAUDrozPDcvJ5M/q8F
# ljUfV1q3/875PbcstvZU3cjnEjpNrkyKt1yatLcgPcp/IjSufjtoZgFE5wFORlOb
# M2D3lL5TN5BzQ/Myw1Pv26r+dE5px2uMYJPexMcM3+EyrsyTO1F4lWeL7j1W/gzQ
# aQ8bD/MlJmszbfduR/pzQ+V+DqVmsSl8MoRjVYnEDcGTVDAZE6zTfTen6106bDVc
# 20HXEtqpSQvf2ICKCZNijrVmzyWIzYS4sT+kOQ/ZAp7rEkyVfPNrBaleFoPMuGfi
# 6BOdzFuC00yz7Vv/3uVzrCM7LQC/NVV0CUnYSVgaf5I25lGSDvMmfRxNF7zJ7EMm
# 0L9BX0CpRET0medXh55QH1dUqD79dGMvsVBlCeZYQi5DGky08CVHWfoEHpPUJkZK
# UIGy3r54t/xnFeHJV4QeD2PW6WK61l9VLupcxigIBCU5uA4rqfJMlxwHPw1S9e3v
# L4IPAgMBAAGjgdAwgc0wEwYDVR0lBAwwCgYIKwYBBQUHAwMwEgYDVR0TAQH/BAgw
# BgEB/wIBAjAdBgNVHQ4EFgQUU3m/WqorSs9UgOHYm8Cd8rIDZsswCwYDVR0PBAQD
# AgGGMB8GA1UdIwQYMBaAFGL7CiFbf0NuEdoJVFBr9dKWcfGeMFUGA1UdHwROMEww
# SqBIoEaGRGh0dHA6Ly9jcmwubWljcm9zb2Z0LmNvbS9wa2kvY3JsL3Byb2R1Y3Rz
# L01pY3Jvc29mdENvZGVWZXJpZlJvb3QuY3JsMA0GCSqGSIb3DQEBBQUAA4ICAQBr
# IpM8PTlUcWRrDvLkPDARxSBKS4YPkvH/M3k62eSYpw5AoCKAfmGy4KcZzyaVMSpl
# 1GpPMYbqwMYuxWSMPUhZzQsvdD2UJhMQQtSXmCdePHbSeGkdGmTnBXJ14OtmQEOf
# jwxG/5dgpshnrRAIm2Km6b46itMHTZ9ykyW8BhHgLJA4Pmcc/RnXnpDOPcLg52Gs
# wOUE9R6ZVAyRDQFWcTeuJ9SeQyKlySfNTeVxEjkkpUFWh/+8VRQPJcqJ7seX5dIT
# /z1+GqCPP8gs16Nw0MdgwPzYPlHnl8Y+O+3PeL6KyuPE8qen7Z6uCAKPoFLbch7V
# O8NNn476m3DH+OO/bD+Sm+Q3PuxqjCn5waK/iz4aaWb7HGNPJgHJAsQ+0v/DQ6gb
# /Zn61LylueKTLzsBxdH0Oi9ow+Bkt1qVXkbMB4NpuzwFklZzNXNFmE582BKlt0Lp
# omP2QmAYcNE7bzHAh8fmceHzRhbp9bhys+ltH2ImSaNJi91ox4toVvfe/PqHJLgD
# gReP5fFnah2u03T3jKVdswuOQimWzknEd35mfAEXGmwUJMOwF3cF2BpAt4Zr2OR7
# QKx+305vJPkggIKMM+fl+inYndqLcF0ryR2CTAtny4RBnucGfhGDRC2KGe70f5rd
# eRw3GR6fP4wpug1cEIY3bEjNRV3NcLy80U1d2MW4djGCFgcwghYDAgEBMIGRMHwx
# CzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNV
# BAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDEkMCIGA1UEAxMb
# U2VjdGlnbyBSU0EgQ29kZSBTaWduaW5nIENBAhEAzS1l4rwsCIvYBjRVawV4ujAN
# BglghkgBZQMEAgEFAKBqMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisG
# AQQBgjcCAQsxDjAMBgorBgEEAYI3AgEWMC8GCSqGSIb3DQEJBDEiBCDIyEEcXKXy
# YUqcKFaHDMb568+ti4l0+yFEPw5wxqhE0zANBgkqhkiG9w0BAQEFAASCAYCJP8db
# Kg5SiEMseOi0pMMLeVzjjuETby+AHJm+PcDr5F+0OGZ5fhO0T/m4960lF0hFsRmx
# iWKYkn9xhXoGORgAyjo3GvomO16bP8CRtgyd1guvaKC7uMVu6t5bvYe3Ng7mQeZd
# JfG/nOCIo4cwk/xuNkkoq8rA3Gx2Z0xcFtIv2XSFomkAcVlgGDJr9jZNm+3t7Rwn
# jgttATslmcKkjzbnGxb/rIBJMdCU6FmA3VQXIJP+HMeYY5uo9GupV6DHRq4Izz9D
# DGv4uziT7BusDbzln8Zsgob0XdKs9nBgHWydXAI6jcD86yvCck3xPDtBC9rXQqbC
# cE+zJTD/d4++IXUi0YhWEE+542z5Yh/Pspug4Ki3NwlbJryXg7r62UGZXtcCOWP9
# lWuwgUf/wE78CCmRlZvGtOLhJHyJxttKw8ucvBd4qfFvU9eIALOCYqeEXvq94zYg
# bFPtdAS83+NrYQc6k81jnzy9TVOhI5KANvkMX0CVyt0EkFTFQkL3JvGy4aehghNa
# MIITVgYKKwYBBAGCNwMDATGCE0YwghNCBgkqhkiG9w0BBwKgghMzMIITLwIBAzEP
# MA0GCWCGSAFlAwQCAgUAMIH5BgsqhkiG9w0BCRABBKCB6QSB5jCB4wIBAQYKKwYB
# BAGyMQIBATAxMA0GCWCGSAFlAwQCAQUABCDlExBgDJaS8/nE5HdK35ERms3DYyk3
# G85+9uGZYQn3iQIUT1Mc+Y1lRPoWAiZN17kpJgHKMVcYDzIwMjIwODMwMjMzNjAw
# WgIITgAhS4quV8CgbqRsMGoxCzAJBgNVBAYTAkdCMRMwEQYDVQQIEwpNYW5jaGVz
# dGVyMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLDAqBgNVBAMMI1NlY3RpZ28g
# UlNBIFRpbWUgU3RhbXBpbmcgU2lnbmVyICMzoIIN6jCCBvYwggTeoAMCAQICEQCQ
# OX+a0ko6E/K9kV8IOKlDMA0GCSqGSIb3DQEBDAUAMH0xCzAJBgNVBAYTAkdCMRsw
# GQYDVQQIExJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcTB1NhbGZvcmQxGDAW
# BgNVBAoTD1NlY3RpZ28gTGltaXRlZDElMCMGA1UEAxMcU2VjdGlnbyBSU0EgVGlt
# ZSBTdGFtcGluZyBDQTAeFw0yMjA1MTEwMDAwMDBaFw0zMzA4MTAyMzU5NTlaMGox
# CzAJBgNVBAYTAkdCMRMwEQYDVQQIEwpNYW5jaGVzdGVyMRgwFgYDVQQKEw9TZWN0
# aWdvIExpbWl0ZWQxLDAqBgNVBAMMI1NlY3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcg
# U2lnbmVyICMzMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAkLJxP3nh
# 1LmKF8zDl8KQlHLtWjpvAUN/c1oonyR8oDVABvqUrwqhg7YT5EsVBl5qiiA0cXu7
# Ja0/WwqkHy9sfS5hUdCMWTc+pl3xHl2AttgfYOPNEmqIH8b+GMuTQ1Z6x84D1gBk
# KFYisUsZ0vCWyUQfOV2csJbtWkmNfnLkQ2t/yaA/bEqt1QBPvQq4g8W9mCwHdgFw
# Rd7D8EJp6v8mzANEHxYo4Wp0tpxF+rY6zpTRH72MZar9/MM86A2cOGbV/H0em1mM
# kVpCV1VQFg1LdHLuoCox/CYCNPlkG1n94zrU6LhBKXQBPw3gE3crETz7Pc3Q5+GX
# W1X3KgNt1c1i2s6cHvzqcH3mfUtozlopYdOgXCWzpSdoo1j99S1ryl9kx2soDNqs
# eEHeku8Pxeyr3y1vGlRRbDOzjVlg59/oFyKjeUFiz/x785LaruA8Tw9azG7fH7wi
# r7c4EJo0pwv//h1epPPuFjgrP6x2lEGdZB36gP0A4f74OtTDXrtpTXKZ5fEyLVH6
# Ya1N6iaObfypSJg+8kYNabG3bvQF20EFxhjAUOT4rf6sY2FHkbxGtUZTbMX04YYn
# k4Q5bHXgHQx6WYsuy/RkLEJH9FRYhTflx2mn0iWLlr/GreC9sTf3H99Ce6rrHOnr
# PVrd+NKQ1UmaOh2DGld/HAHCzhx9zPuWFcUCAwEAAaOCAYIwggF+MB8GA1UdIwQY
# MBaAFBqh+GEZIA/DQXdFKI7RNV8GEgRVMB0GA1UdDgQWBBQlLmg8a5orJBSpH6Lf
# JjrPFKbx4DAOBgNVHQ8BAf8EBAMCBsAwDAYDVR0TAQH/BAIwADAWBgNVHSUBAf8E
# DDAKBggrBgEFBQcDCDBKBgNVHSAEQzBBMDUGDCsGAQQBsjEBAgEDCDAlMCMGCCsG
# AQUFBwIBFhdodHRwczovL3NlY3RpZ28uY29tL0NQUzAIBgZngQwBBAIwRAYDVR0f
# BD0wOzA5oDegNYYzaHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdvUlNBVGlt
# ZVN0YW1waW5nQ0EuY3JsMHQGCCsGAQUFBwEBBGgwZjA/BggrBgEFBQcwAoYzaHR0
# cDovL2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUlNBVGltZVN0YW1waW5nQ0EuY3J0
# MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG9w0B
# AQwFAAOCAgEAc9rtaHLLwrlAoTG7tAOjLRR7JOe0WxV9qOn9rdGSDXw9NqBp2fOa
# MNqsadZ0VyQ/fg882fXDeSVsJuiNaJPO8XeJOX+oBAXaNMMU6p8IVKv/xH6WbCvT
# lOu0bOBFTSyy9zs7WrXB+9eJdW2YcnL29wco89Oy0OsZvhUseO/NRaAA5PgEdrtX
# xZC+d1SQdJ4LT03EqhOPl68BNSvLmxF46fL5iQQ8TuOCEmLrtEQMdUHCDzS4iJ3I
# IvETatsYL254rcQFtOiECJMH+X2D/miYNOR35bHOjJRs2wNtKAVHfpsu8GT726QD
# MRB8Gvs8GYDRC3C5VV9HvjlkzrfaI1Qy40ayMtjSKYbJFV2Ala8C+7TRLp04fDXg
# DxztG0dInCJqVYLZ8roIZQPl8SnzSIoJAUymefKithqZlOuXKOG+fRuhfO1WgKb0
# IjOQ5IRT/Cr6wKeXqOq1jXrO5OBLoTOrC3ag1WkWt45mv1/6H8Sof6ehSBSRDYL8
# vU2Z7cnmbDb+d0OZuGktfGEv7aOwSf5bvmkkkf+T/FdpkkvZBT9thnLTotDAZNI6
# QsEaA/vQ7ZohuD+vprJRVNVMxcofEo1XxjntXP/snyZ2rWRmZ+iqMODSrbd9sWpB
# J24DiqN04IoJgm6/4/a3vJ4LKRhogaGcP24WWUsUCQma5q6/YBXdhvUwggbsMIIE
# 1KADAgECAhAwD2+s3WaYdHypRjaneC25MA0GCSqGSIb3DQEBDAUAMIGIMQswCQYD
# VQQGEwJVUzETMBEGA1UECBMKTmV3IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5IENp
# dHkxHjAcBgNVBAoTFVRoZSBVU0VSVFJVU1QgTmV0d29yazEuMCwGA1UEAxMlVVNF
# UlRydXN0IFJTQSBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eTAeFw0xOTA1MDIwMDAw
# MDBaFw0zODAxMTgyMzU5NTlaMH0xCzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVh
# dGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3Rp
# Z28gTGltaXRlZDElMCMGA1UEAxMcU2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBD
# QTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAMgbAa/ZLH6ImX0BmD8g
# kL2cgCFUk7nPoD5T77NawHbWGgSlzkeDtevEzEk0y/NFZbn5p2QWJgn71TJSeS7J
# Y8ITm7aGPwEFkmZvIavVcRB5h/RGKs3EWsnb111JTXJWD9zJ41OYOioe/M5YSdO/
# 8zm7uaQjQqzQFcN/nqJc1zjxFrJw06PE37PFcqwuCnf8DZRSt/wflXMkPQEovA8N
# T7ORAY5unSd1VdEXOzQhe5cBlK9/gM/REQpXhMl/VuC9RpyCvpSdv7QgsGB+uE31
# DT/b0OqFjIpWcdEtlEzIjDzTFKKcvSb/01Mgx2Bpm1gKVPQF5/0xrPnIhRfHuCkZ
# pCkvRuPd25Ffnz82Pg4wZytGtzWvlr7aTGDMqLufDRTUGMQwmHSCIc9iVrUhcxIe
# /arKCFiHd6QV6xlV/9A5VC0m7kUaOm/N14Tw1/AoxU9kgwLU++Le8bwCKPRt2ieK
# BtKWh97oaw7wW33pdmmTIBxKlyx3GSuTlZicl57rjsF4VsZEJd8GEpoGLZ8DXv2D
# olNnyrH6jaFkyYiSWcuoRsDJ8qb/fVfbEnb6ikEk1Bv8cqUUotStQxykSYtBORQD
# Hin6G6UirqXDTYLQjdprt9v3GEBXc/Bxo/tKfUU2wfeNgvq5yQ1TgH36tjlYMu9v
# GFCJ10+dM70atZ2h3pVBeqeDAgMBAAGjggFaMIIBVjAfBgNVHSMEGDAWgBRTeb9a
# qitKz1SA4dibwJ3ysgNmyzAdBgNVHQ4EFgQUGqH4YRkgD8NBd0UojtE1XwYSBFUw
# DgYDVR0PAQH/BAQDAgGGMBIGA1UdEwEB/wQIMAYBAf8CAQAwEwYDVR0lBAwwCgYI
# KwYBBQUHAwgwEQYDVR0gBAowCDAGBgRVHSAAMFAGA1UdHwRJMEcwRaBDoEGGP2h0
# dHA6Ly9jcmwudXNlcnRydXN0LmNvbS9VU0VSVHJ1c3RSU0FDZXJ0aWZpY2F0aW9u
# QXV0aG9yaXR5LmNybDB2BggrBgEFBQcBAQRqMGgwPwYIKwYBBQUHMAKGM2h0dHA6
# Ly9jcnQudXNlcnRydXN0LmNvbS9VU0VSVHJ1c3RSU0FBZGRUcnVzdENBLmNydDAl
# BggrBgEFBQcwAYYZaHR0cDovL29jc3AudXNlcnRydXN0LmNvbTANBgkqhkiG9w0B
# AQwFAAOCAgEAbVSBpTNdFuG1U4GRdd8DejILLSWEEbKw2yp9KgX1vDsn9FqguUlZ
# kClsYcu1UNviffmfAO9Aw63T4uRW+VhBz/FC5RB9/7B0H4/GXAn5M17qoBwmWFzz
# tBEP1dXD4rzVWHi/SHbhRGdtj7BDEA+N5Pk4Yr8TAcWFo0zFzLJTMJWk1vSWVgi4
# zVx/AZa+clJqO0I3fBZ4OZOTlJux3LJtQW1nzclvkD1/RXLBGyPWwlWEZuSzxWYG
# 9vPWS16toytCiiGS/qhvWiVwYoFzY16gu9jc10rTPa+DBjgSHSSHLeT8AtY+dwS8
# BDa153fLnC6NIxi5o8JHHfBd1qFzVwVomqfJN2Udvuq82EKDQwWli6YJ/9GhlKZO
# qj0J9QVst9JkWtgqIsJLnfE5XkzeSD2bNJaaCV+O/fexUpHOP4n2HKG1qXUfcb9b
# Q11lPVCBbqvw0NP8srMftpmWJvQ8eYtcZMzN7iea5aDADHKHwW5NWtMe6vBE5jJv
# HOsXTpTDeGUgOw9Bqh/poUGd/rG4oGUqNODeqPk85sEwu8CgYyz8XBYAqNDEf+oR
# nR4GxqZtMl20OAkrSQeq/eww2vGnL8+3/frQo4TZJ577AWZ3uVYQ4SBuxq6x+ba6
# yDVdM3aO8XwgDCp3rrWiAoa6Ke60WgCxjKvj+QrJVF3UuWp0nr1IrpgxggQtMIIE
# KQIBATCBkjB9MQswCQYDVQQGEwJHQjEbMBkGA1UECBMSR3JlYXRlciBNYW5jaGVz
# dGVyMRAwDgYDVQQHEwdTYWxmb3JkMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQx
# JTAjBgNVBAMTHFNlY3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcgQ0ECEQCQOX+a0ko6
# E/K9kV8IOKlDMA0GCWCGSAFlAwQCAgUAoIIBazAaBgkqhkiG9w0BCQMxDQYLKoZI
# hvcNAQkQAQQwHAYJKoZIhvcNAQkFMQ8XDTIyMDgzMDIzMzYwMFowPwYJKoZIhvcN
# AQkEMTIEMDiFcSdEzpkoCQMCZyhUFPUXxmsLFn/5XAOk7Gb8nBEBMF6bpPpxkxO/
# lnWtfgTVrzCB7QYLKoZIhvcNAQkQAgwxgd0wgdowgdcwFgQUqzQBOqxAlzGfCBrw
# sxjhg/gPeIEwgbwEFALWW5Xig3DBVwCV+oj5I92Tf62PMIGjMIGOpIGLMIGIMQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKTmV3IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5
# IENpdHkxHjAcBgNVBAoTFVRoZSBVU0VSVFJVU1QgTmV0d29yazEuMCwGA1UEAxMl
# VVNFUlRydXN0IFJTQSBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eQIQMA9vrN1mmHR8
# qUY2p3gtuTANBgkqhkiG9w0BAQEFAASCAgAZbNYGTXcnvlKEIgAgmh1iIPpIfCgU
# 00z5Nwue+8YKMQPWdEFCaOe75ddr64dKa58WHCplG2k90AaogA063/krYQ4FEOTu
# TPfhot932KbG3hS4yXO3as7Lk14J2bxY5+Q2XsXwxhVGPWqGasynUtmLKnfbiHhs
# LjJ+yQURwo0evLU8HvdlMIZff3GQny0hhMrYWJlQdOva09O9lGiMSEUZbWM/2pci
# U6hCVSo1wOR3iSy4gX3E8qczZUKxon6/Dgh0teOfKI/Yg23wuPTDKVOxF69Fe7eI
# zfViROg4i/pjXlaHcTrcJafV8AyjIH7EASCGlYBBlWGY8oDXdKr/AOQpOHqWR0Lw
# 8CgVIp/Ka+jg9iIjrLvl5lTTEfnhf2s7FeiwZoIXw4AS3MXgp4IB+LcxjXWeSYVo
# Ot9qlZ6LWebp2IEuIDnowHat2EGDAUmD9yiFffEvCvKOHK0QrsufuMvUPjtqiJLp
# PRjSQjtqfb7lEY5koNaBnK99CtmvnpZJyGeeHwtBNnSib/4UlYe+oVIjgumWa1VP
# cKhg6U8ERAeliHT+thISUxK8VNMuhY1u+PK3TRYxy4LQqYlKizes2pUFIVbnmWBs
# tZsAB/u5f8BHsGgTk/jLJx5zi/AgD+smyDMMySZNEhVWGzivPGvWW7YCUfIv/il9
# n52Z3xslAi9WXA==
# SIG # End signature block
