# <copyright>
# INTEL CONFIDENTIAL
#
# Copyright 2021 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of
# them is governed by the express license under which they were provided to you ("License").
# Unless the License provides otherwise, you may not use, modify, copy, publish, distribute,
# disclose or transmit this software or the related documents without Intel's prior written
# permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#
# <copyright>

# Functions common to all cmdlets
Function ValidateGetAdapterNameParams($AdapterName, $Adapters, [ref]$ErrorMessages)
{
    $AdapterNames = @()

    if ((-Not $Adapters) -and $AdapterName)
    {
        foreach($n in $AdapterName)
        {
            $TmpAdapters = ($script:PnpDevice.Where({$_.Name -Like $n})).Name
            if (-Not $TmpAdapters)
            {
                $ErrorMessages.Value += $Messages.AdapterNotFound -f $n
                continue
            }
            $AdapterNames += $TmpAdapters
        }
    }
    elseif ((-Not $AdapterName) -and $Adapters)
    {
        foreach ($a in $Adapters)
        {
            if ($a.CreationClassName -eq "MSFT_NetAdapter")
            {
                $AdapterNames += $a.ifDesc
            }
            elseif ($a.CreationClassName -eq "Win32_NetworkAdapter" -or
                    $a.CreationClassName -eq "IANet_PhysicalEthernetAdapter" -or
                    $a.CreationClassName -eq "Win32_PnpEntity" -or
                    ($null -ne $a.PSTypeNames -and $a.PSTypeNames[0] -eq "IntelEthernetAdapter"))
            {
                $AdapterNames += $a.Name
            }
            else
            {
                $ErrorMessages.Value += $Messages.InvalidObject -f $a.GetType().Name
            }
        }
    }
    elseif (-Not ($AdapterName -and $Adapters))
    {
        $AdapterNames = ($script:PnpDevice.Where({$_.Manufacturer -eq "Intel"})).Name
    }
    elseif ($AdapterName -and $Adapters)
    {
        $ErrorMessages.Value += $Messages.InvalidParamsAdapterAndName
    }

    return $AdapterNames
}

Function ValidateSetAdapterNameParams($AdapterName, $Adapters, [ref]$ErrorMessages)
{
    $AdapterNames = @()

    do
    {
        if ($AdapterName -and $Adapters)
        {
            $ErrorMessages.Value += $Messages.InvalidParamsAdapterAndName
            break
        }
        elseif ($AdapterName)
        {
            foreach ($n in $AdapterName)
            {
                $TmpAdapters = ($script:PnpDevice.Where({$_.Name -Like $n})).Name
                if (-Not $TmpAdapters)
                {
                    $ErrorMessages.Value += $Messages.AdapterNotFound -f $n
                    continue
                }
                $AdapterNames += $TmpAdapters
            }
        }
        elseif ($Adapters)
        {
            foreach ($a in $Adapters)
            {
                if ($a.CreationClassName -eq "MSFT_NetAdapter")
                {
                    $AdapterNames += $a.ifDesc
                }
                elseif ($a.CreationClassName -eq "Win32_NetworkAdapter" -or
                        $a.CreationClassName -eq "IANet_PhysicalEthernetAdapter" -or
                        $a.CreationClassName -eq "Win32_PnpEntity" -or
                        ($null -ne $a.PSTypeNames -and $a.PSTypeNames[0] -eq "IntelEthernetAdapter"))
                {
                    $AdapterNames += $a.Name
                }
                else
                {
                    $ErrorMessages.Value += $Messages.InvalidObject -f $a.GetType().Name
                }
            }
        }
        else
        {
            $ErrorMessages.Value += $Messages.InvalidParamsAdapterOrName
            break
        }
    } while ($false)

    return $AdapterNames
}

Function ValidatePathParams([ref]$LogPath, $UseDefaultPath, $LogName, [ref]$ErrorMessages)
{
    $Result = $true

    try
    {
        if ($UseDefaultPath)
        {
            $DefaultPath = $ENV:LOCALAPPDATA + "\Intel\Wired Networking\" + $LogName

            if (-not (Test-Path -Path $DefaultPath -ErrorAction Stop))
            {
                New-Item -Path $ENV:LOCALAPPDATA -Name "\Intel\Wired Networking" -ItemType "directory" -ErrorAction SilentlyContinue
            }
            $LogPath.Value = $DefaultPath
        }
        else
        {
            $LogPath.Value = $Path
        }

        $isPathFile = Test-Path -Path $LogPath.Value -PathType Leaf -ErrorAction Stop

        if (($isPathFile) -and (-not $Append) -and (-not $Force))
        {
            $ErrorMessages.Value += $Messages.LogmanFileExists -f $AdapterName
            $Result = $false
        }
        elseif (-not $isPathFile)
        {
            if (Test-Path -Path $LogPath.Value -ErrorAction Stop)
            {
                $ErrorMessages.Value += $Messages.FolderFileNameExits
                $Result = $false
            }
            else
            {
                $AbsolutePath = [IO.Path]::GetFullPath($LogPath.Value)
                $ParentFolder = Split-Path -Path $AbsolutePath

                if (-Not (Test-Path -Path $ParentFolder -ErrorAction Stop))
                {
                    $ErrorMessages.Value += $Messages.PathIncorrect
                    $Result = $false
                }
            }
        }
    }
    catch
    {
        $ErrorMessages.Value += $Messages.PathIncorrect
        $Result = $false
    }
    return $Result
}


function InvokeCimMethod($ClassName, $InstanceName = "", $MethodName, $params = @{}, $Namespace = "root\wmi")
{
    $query = "Select * from $ClassName"
    if ($InstanceName)
    {
        $query+= " where instancename like '$InstanceName'"
    }

    Invoke-CimMethod -Query $query -MethodName $MethodName -Arguments $params -Namespace $Namespace -ErrorAction SilentlyContinue
}

function GetIntelEthernetDevices()
{
    $script:PnpDevice = Get-PnpDevice | Where-Object {$_.Class -eq "Net"} -ErrorAction SilentlyContinue
    $script:SupportedAdapters = $script:PnpDevice.Where({$_.Service -in @('icea', 'scea', 'i40ea', 'i40eb')})
}

function GetSupportedAdapters($AdapterNames, [ref]$WarningMessages)
{
    $SupportedAdapterNames = @()
    foreach ($a in $AdapterNames)
    {
        if ($script:SupportedAdapters.Name -Contains $a)
        {
            $SupportedAdapterNames += $a
        }
        else
        {
            $WarningMessages.Value += $Messages.NoCmdletSupport -f $a
        }
    }

    return $SupportedAdapterNames
}

function CheckDeviceError($AdapterName)
{
    $SupportedAdapter = $script:SupportedAdapters.Where({$_.FriendlyName -eq $AdapterName})
    # if the device is not 'working properly'
    if ([Int32]$SupportedAdapter.ConfigManagerErrorCode -ne 0)
    {
        # Workaround due to ProblemDescription being empty by default - change current path
        Push-Location -Path (Get-Module -Name PnPDevice).ModuleBase
        $StatusMsg = $AdapterName + ": " + $SupportedAdapter.ProblemDescription
        # Reset path
        Pop-Location
        $StatusMsg
    }
}

function ValidateSingleAdapter($PipelineInput, $AdapterName, [ref]$ErrorMessages)
{
    $Result = $false

    do
    {
        if ($PipelineInput.Count -gt 1)
        {
            $ErrorMessages.Value += $Messages.InvalidParams
            break
        }

        if ($AdapterName.Count -gt 1)
        {
            $ErrorMessages.Value += $Messages.InvalidParams
            break
        }

        $Result = $true
    } while ($false)

    return $Result
}

# SIG # Begin signature block
# MIIotgYJKoZIhvcNAQcCoIIopzCCKKMCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCDKqs8zWcPyF5Tf
# G+rx978gIjg4ykf8qrZLK4Sxbv9rhaCCEgUwggWeMIIEhqADAgECAhEAzS1l4rws
# CIvYBjRVawV4ujANBgkqhkiG9w0BAQsFADB8MQswCQYDVQQGEwJHQjEbMBkGA1UE
# CBMSR3JlYXRlciBNYW5jaGVzdGVyMRAwDgYDVQQHEwdTYWxmb3JkMRgwFgYDVQQK
# Ew9TZWN0aWdvIExpbWl0ZWQxJDAiBgNVBAMTG1NlY3RpZ28gUlNBIENvZGUgU2ln
# bmluZyBDQTAeFw0yMTA0MDIwMDAwMDBaFw0yMzA0MDIyMzU5NTlaMIGEMQswCQYD
# VQQGEwJVUzETMBEGA1UECAwKQ2FsaWZvcm5pYTEUMBIGA1UEBwwLU2FudGEgQ2xh
# cmExGjAYBgNVBAoMEUludGVsIENvcnBvcmF0aW9uMRIwEAYDVQQLDAlTb2xhcktN
# Q1MxGjAYBgNVBAMMEUludGVsIENvcnBvcmF0aW9uMIIBojANBgkqhkiG9w0BAQEF
# AAOCAY8AMIIBigKCAYEA7CCN9iKpDHOrRceKhlXFP/tf6Lllw2H2fR9KVI4/fQIx
# MU1hXwnlHmAzMCY7IgcCFY4p3F5/MJGKaqYngwOo28Zo6Q1N6ukysA7PSavmF2RY
# WD6VFeya/2H0PoNeRFjHaRzSeynFFeJAFew9r7UReUwM/507sxZYPQuWWIdAEK7H
# Dqp2VlHmgZOXVGHhNO6GFOKpC/C01g6X3x6OquddRNMt5UrZzZzDo5MpJz9SBB2V
# jiqwZ80dvNR2W2xi90cIHh4BkXvB54UNkp4VTVu16T0k3cweo+C39U7GrCAr5Axz
# DETjBvhNtP1sf9SoRV7xY6g5wssfI7yYT9J0gsifn/Vy8MWH355TPoA+PVhbAu0m
# 9FMz4EWu55nnUurNML2jaUxsos21/7ELat12kWC0tq9fhkODjKO8X9PuiBHflZLk
# d3F4QcSMvuGocWGqE77VV3vn8jlvigm2TOV0CfGTQajGMX0jeTRZ19fzBNkt2X9d
# SSGolI/Kj1gSvCggpkUBAgMBAAGjggGQMIIBjDAfBgNVHSMEGDAWgBQO4TqoUzox
# 1Yq+wbutZxoDha00DjAdBgNVHQ4EFgQUshkNuM2SdwJnW4vFy8c4FtUTrbQwDgYD
# VR0PAQH/BAQDAgeAMAwGA1UdEwEB/wQCMAAwEwYDVR0lBAwwCgYIKwYBBQUHAwMw
# EQYJYIZIAYb4QgEBBAQDAgQQMEoGA1UdIARDMEEwNQYMKwYBBAGyMQECAQMCMCUw
# IwYIKwYBBQUHAgEWF2h0dHBzOi8vc2VjdGlnby5jb20vQ1BTMAgGBmeBDAEEATBD
# BgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29S
# U0FDb2RlU2lnbmluZ0NBLmNybDBzBggrBgEFBQcBAQRnMGUwPgYIKwYBBQUHMAKG
# Mmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1JTQUNvZGVTaWduaW5nQ0Eu
# Y3J0MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG
# 9w0BAQsFAAOCAQEAVadLNRW4f/pKMqrbn0BdOoQ8/1EJ87gvVfosei2bLwTEvpmv
# mn2n561H6AFedtIJ6L4FmXII4M4r20i+5LREbI6PpKDmOAf4xW7POxfCRvkTQAZO
# 3zoVxjMQBXo7cZVF1xHCdviXzD1usuIiCF8DLm6z4O/kyeFFNcn816yPQct91Pnk
# SBBVvL+Kwu8xvR+ZIQy632WUA4HnNpRdFnVSzUifEg2GrtsKZR8k+rm2o8K8yjJq
# 3SznwgJQCMVMh3CtRtUwE/c7o/6rvm53fTYJDd3aoPHVgH6S2WqS3+3mQG7A6hTD
# nrP/mYnS4PF7XzxxjZhUlhy4G/MarJPvT9IrNDCCBfUwggPdoAMCAQICEB2iSDBv
# myYY0ILgln0z02owDQYJKoZIhvcNAQEMBQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpOZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkgQ2l0eTEeMBwGA1UEChMV
# VGhlIFVTRVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVVU0VSVHJ1c3QgUlNBIENl
# cnRpZmljYXRpb24gQXV0aG9yaXR5MB4XDTE4MTEwMjAwMDAwMFoXDTMwMTIzMTIz
# NTk1OVowfDELMAkGA1UEBhMCR0IxGzAZBgNVBAgTEkdyZWF0ZXIgTWFuY2hlc3Rl
# cjEQMA4GA1UEBxMHU2FsZm9yZDEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSQw
# IgYDVQQDExtTZWN0aWdvIFJTQSBDb2RlIFNpZ25pbmcgQ0EwggEiMA0GCSqGSIb3
# DQEBAQUAA4IBDwAwggEKAoIBAQCGIo0yhXoYn0nwli9jCB4t3HyfFM/jJrYlZilA
# hlRGdDFixRDtsocnppnLlTDAVvWkdcapDlBipVGREGrgS2Ku/fD4GKyn/+4uMyD6
# DBmJqGx7rQDDYaHcaWVtH24nlteXUYam9CflfGqLlR5bYNV+1xaSnAAvaPeX7Wpy
# vjg7Y96Pv25MQV0SIAhZ6DnNj9LWzwa0VwW2TqE+V2sfmLzEYtYbC43HZhtKn52B
# xHJAteJf7wtF/6POF6YtVbC3sLxUap28jVZTxvC6eVBJLPcDuf4vZTXyIuosB69G
# 2flGHNyMfHEo8/6nxhTdVZFuihEN3wYklX0Pp6F8OtqGNWHTAgMBAAGjggFkMIIB
# YDAfBgNVHSMEGDAWgBRTeb9aqitKz1SA4dibwJ3ysgNmyzAdBgNVHQ4EFgQUDuE6
# qFM6MdWKvsG7rWcaA4WtNA4wDgYDVR0PAQH/BAQDAgGGMBIGA1UdEwEB/wQIMAYB
# Af8CAQAwHQYDVR0lBBYwFAYIKwYBBQUHAwMGCCsGAQUFBwMIMBEGA1UdIAQKMAgw
# BgYEVR0gADBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8vY3JsLnVzZXJ0cnVzdC5j
# b20vVVNFUlRydXN0UlNBQ2VydGlmaWNhdGlvbkF1dGhvcml0eS5jcmwwdgYIKwYB
# BQUHAQEEajBoMD8GCCsGAQUFBzAChjNodHRwOi8vY3J0LnVzZXJ0cnVzdC5jb20v
# VVNFUlRydXN0UlNBQWRkVHJ1c3RDQS5jcnQwJQYIKwYBBQUHMAGGGWh0dHA6Ly9v
# Y3NwLnVzZXJ0cnVzdC5jb20wDQYJKoZIhvcNAQEMBQADggIBAE1jUO1HNEphpNve
# aiqMm/EAAB4dYns61zLC9rPgY7P7YQCImhttEAcET7646ol4IusPRuzzRl5ARokS
# 9At3WpwqQTr81vTr5/cVlTPDoYMot94v5JT3hTODLUpASL+awk9KsY8k9LOBN9O3
# ZLCmI2pZaFJCX/8E6+F0ZXkI9amT3mtxQJmWunjxucjiwwgWsatjWsgVgG10Xkp1
# fqW4w2y1z99KeYdcx0BNYzX2MNPPtQoOCwR/oEuuu6Ol0IQAkz5TXTSlADVpbL6f
# ICUQDRn7UJBhvjmPeo5N9p8OHv4HURJmgyYZSJXOSsnBf/M6BZv5b9+If8AjntIe
# Q3pFMcGcTanwWbJZGehqjSkEAnd8S0vNcL46slVaeD68u28DECV3FTSK+TbMQ5Lk
# uk/xYpMoJVcp+1EZx6ElQGqEV8aynbG8HArafGd+fS7pKEwYfsR7MUFxmksp7As9
# V1DSyt39ngVR5UR43QHesXWYDVQk/fBO4+L4g71yuss9Ou7wXheSaG3IYfmm8SoK
# C6W59J7umDIFhZ7r+YMp08Ysfb06dy6LN0KgaoLtO0qqlBCk4Q34F8W2WnkzGJLj
# tXX4oemOCiUe5B7xn1qHI/+fpFGe+zmAEc3btcSnqIBv5VPU4OOiwtJbGvoyJi1q
# V3AcPKRYLqPzW0sH3DJZ84enGm1YMIIGZjCCBE6gAwIBAgITMwAAAES3P/zvWs+i
# egAAAAAARDANBgkqhkiG9w0BAQUFADB/MQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMSkwJwYDVQQDEyBNaWNyb3NvZnQgQ29kZSBWZXJpZmljYXRp
# b24gUm9vdDAeFw0xNTA3MjIyMTAzNDlaFw0yNTA3MjIyMTAzNDlaMIGIMQswCQYD
# VQQGEwJVUzETMBEGA1UECBMKTmV3IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5IENp
# dHkxHjAcBgNVBAoTFVRoZSBVU0VSVFJVU1QgTmV0d29yazEuMCwGA1UEAxMlVVNF
# UlRydXN0IFJTQSBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eTCCAiIwDQYJKoZIhvcN
# AQEBBQADggIPADCCAgoCggIBAIASZRc2DsPbCLPQrFcNdu3NJ9NMrVCDYeKqIE0J
# LWQJ3M6Jn8w9qez2z8Hc8dOx1ns3KBErR9o5xrw6GbRfpr19naNjQrZ28qk7K5H4
# 4m/Q7BYgkAk+4uh0yRi0kdRiZNt/owbxiBhqkCI8vP4T8IcUe/bkH47U5FHGEWdG
# CFHLhhRUP7wz/n5snP8WnRi9UY41pqdmyHJn2yFmsdSbeAPAUDrozPDcvJ5M/q8F
# ljUfV1q3/875PbcstvZU3cjnEjpNrkyKt1yatLcgPcp/IjSufjtoZgFE5wFORlOb
# M2D3lL5TN5BzQ/Myw1Pv26r+dE5px2uMYJPexMcM3+EyrsyTO1F4lWeL7j1W/gzQ
# aQ8bD/MlJmszbfduR/pzQ+V+DqVmsSl8MoRjVYnEDcGTVDAZE6zTfTen6106bDVc
# 20HXEtqpSQvf2ICKCZNijrVmzyWIzYS4sT+kOQ/ZAp7rEkyVfPNrBaleFoPMuGfi
# 6BOdzFuC00yz7Vv/3uVzrCM7LQC/NVV0CUnYSVgaf5I25lGSDvMmfRxNF7zJ7EMm
# 0L9BX0CpRET0medXh55QH1dUqD79dGMvsVBlCeZYQi5DGky08CVHWfoEHpPUJkZK
# UIGy3r54t/xnFeHJV4QeD2PW6WK61l9VLupcxigIBCU5uA4rqfJMlxwHPw1S9e3v
# L4IPAgMBAAGjgdAwgc0wEwYDVR0lBAwwCgYIKwYBBQUHAwMwEgYDVR0TAQH/BAgw
# BgEB/wIBAjAdBgNVHQ4EFgQUU3m/WqorSs9UgOHYm8Cd8rIDZsswCwYDVR0PBAQD
# AgGGMB8GA1UdIwQYMBaAFGL7CiFbf0NuEdoJVFBr9dKWcfGeMFUGA1UdHwROMEww
# SqBIoEaGRGh0dHA6Ly9jcmwubWljcm9zb2Z0LmNvbS9wa2kvY3JsL3Byb2R1Y3Rz
# L01pY3Jvc29mdENvZGVWZXJpZlJvb3QuY3JsMA0GCSqGSIb3DQEBBQUAA4ICAQBr
# IpM8PTlUcWRrDvLkPDARxSBKS4YPkvH/M3k62eSYpw5AoCKAfmGy4KcZzyaVMSpl
# 1GpPMYbqwMYuxWSMPUhZzQsvdD2UJhMQQtSXmCdePHbSeGkdGmTnBXJ14OtmQEOf
# jwxG/5dgpshnrRAIm2Km6b46itMHTZ9ykyW8BhHgLJA4Pmcc/RnXnpDOPcLg52Gs
# wOUE9R6ZVAyRDQFWcTeuJ9SeQyKlySfNTeVxEjkkpUFWh/+8VRQPJcqJ7seX5dIT
# /z1+GqCPP8gs16Nw0MdgwPzYPlHnl8Y+O+3PeL6KyuPE8qen7Z6uCAKPoFLbch7V
# O8NNn476m3DH+OO/bD+Sm+Q3PuxqjCn5waK/iz4aaWb7HGNPJgHJAsQ+0v/DQ6gb
# /Zn61LylueKTLzsBxdH0Oi9ow+Bkt1qVXkbMB4NpuzwFklZzNXNFmE582BKlt0Lp
# omP2QmAYcNE7bzHAh8fmceHzRhbp9bhys+ltH2ImSaNJi91ox4toVvfe/PqHJLgD
# gReP5fFnah2u03T3jKVdswuOQimWzknEd35mfAEXGmwUJMOwF3cF2BpAt4Zr2OR7
# QKx+305vJPkggIKMM+fl+inYndqLcF0ryR2CTAtny4RBnucGfhGDRC2KGe70f5rd
# eRw3GR6fP4wpug1cEIY3bEjNRV3NcLy80U1d2MW4djGCFgcwghYDAgEBMIGRMHwx
# CzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNV
# BAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDEkMCIGA1UEAxMb
# U2VjdGlnbyBSU0EgQ29kZSBTaWduaW5nIENBAhEAzS1l4rwsCIvYBjRVawV4ujAN
# BglghkgBZQMEAgEFAKBqMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisG
# AQQBgjcCAQsxDjAMBgorBgEEAYI3AgEWMC8GCSqGSIb3DQEJBDEiBCB4wQrCOSlk
# fo0seTMgCC+aAayn9AsOxDCOwhKEUQaLQzANBgkqhkiG9w0BAQEFAASCAYCr63A6
# 59udxYkgfNPssuoEcrmNeUUW+C08PFCOgmS+Gc/3HlYTYYLXHY/7RMNQg+Wp/mly
# b/INnxdMw8LZLi7RxGA9OiKZJ6fTFp7/kNlfV5lF82FbcLjfivlAG37S6mXB0wdn
# NbHR1DBjNJgyRvnnPjXjAuo2Avf2zutpCXZvcqHxc/PsNstPJG59fylSDj8dgoW+
# Yuef7+znSp30mqO60o6HdDywxRW9PIxIbKWOKg4qYLevXV9BcB0/Og5CGDBFJbd4
# a8sURodRuDLr9c1HDrsd1/fpdrD4JxEeVlefkrrbEkbHqB1NeXvnUVMms5g0JXP2
# snyq1dhq937rXQ61pRB95/5ru50WMBtfxRsAHrLKY91RGuFVuzHRm1PUHS77R6A9
# 3ZWHMHGL4gCVuQMG8JBegw5Fqjh7ZeLWVSHkzkS/tKx/a66Vtl/V79kwA5x05JMN
# wsDwYsbOpbnZeeqVX0dkzSp/g+dmjXq77PCKF59DHydjsL92fG7m2fXyLxChghNa
# MIITVgYKKwYBBAGCNwMDATGCE0YwghNCBgkqhkiG9w0BBwKgghMzMIITLwIBAzEP
# MA0GCWCGSAFlAwQCAgUAMIH5BgsqhkiG9w0BCRABBKCB6QSB5jCB4wIBAQYKKwYB
# BAGyMQIBATAxMA0GCWCGSAFlAwQCAQUABCBRGS1FrzZFoFrTDlMKpvJ2kA2hGGja
# 6FqUnWK1ZZ/PQgIUcGOoomEO5asr+/jexyT/v3YhAM4YDzIwMjIxMDA1MTc0NTU1
# WgIIbZQo8ndFE5WgbqRsMGoxCzAJBgNVBAYTAkdCMRMwEQYDVQQIEwpNYW5jaGVz
# dGVyMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLDAqBgNVBAMMI1NlY3RpZ28g
# UlNBIFRpbWUgU3RhbXBpbmcgU2lnbmVyICMzoIIN6jCCBvYwggTeoAMCAQICEQCQ
# OX+a0ko6E/K9kV8IOKlDMA0GCSqGSIb3DQEBDAUAMH0xCzAJBgNVBAYTAkdCMRsw
# GQYDVQQIExJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcTB1NhbGZvcmQxGDAW
# BgNVBAoTD1NlY3RpZ28gTGltaXRlZDElMCMGA1UEAxMcU2VjdGlnbyBSU0EgVGlt
# ZSBTdGFtcGluZyBDQTAeFw0yMjA1MTEwMDAwMDBaFw0zMzA4MTAyMzU5NTlaMGox
# CzAJBgNVBAYTAkdCMRMwEQYDVQQIEwpNYW5jaGVzdGVyMRgwFgYDVQQKEw9TZWN0
# aWdvIExpbWl0ZWQxLDAqBgNVBAMMI1NlY3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcg
# U2lnbmVyICMzMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAkLJxP3nh
# 1LmKF8zDl8KQlHLtWjpvAUN/c1oonyR8oDVABvqUrwqhg7YT5EsVBl5qiiA0cXu7
# Ja0/WwqkHy9sfS5hUdCMWTc+pl3xHl2AttgfYOPNEmqIH8b+GMuTQ1Z6x84D1gBk
# KFYisUsZ0vCWyUQfOV2csJbtWkmNfnLkQ2t/yaA/bEqt1QBPvQq4g8W9mCwHdgFw
# Rd7D8EJp6v8mzANEHxYo4Wp0tpxF+rY6zpTRH72MZar9/MM86A2cOGbV/H0em1mM
# kVpCV1VQFg1LdHLuoCox/CYCNPlkG1n94zrU6LhBKXQBPw3gE3crETz7Pc3Q5+GX
# W1X3KgNt1c1i2s6cHvzqcH3mfUtozlopYdOgXCWzpSdoo1j99S1ryl9kx2soDNqs
# eEHeku8Pxeyr3y1vGlRRbDOzjVlg59/oFyKjeUFiz/x785LaruA8Tw9azG7fH7wi
# r7c4EJo0pwv//h1epPPuFjgrP6x2lEGdZB36gP0A4f74OtTDXrtpTXKZ5fEyLVH6
# Ya1N6iaObfypSJg+8kYNabG3bvQF20EFxhjAUOT4rf6sY2FHkbxGtUZTbMX04YYn
# k4Q5bHXgHQx6WYsuy/RkLEJH9FRYhTflx2mn0iWLlr/GreC9sTf3H99Ce6rrHOnr
# PVrd+NKQ1UmaOh2DGld/HAHCzhx9zPuWFcUCAwEAAaOCAYIwggF+MB8GA1UdIwQY
# MBaAFBqh+GEZIA/DQXdFKI7RNV8GEgRVMB0GA1UdDgQWBBQlLmg8a5orJBSpH6Lf
# JjrPFKbx4DAOBgNVHQ8BAf8EBAMCBsAwDAYDVR0TAQH/BAIwADAWBgNVHSUBAf8E
# DDAKBggrBgEFBQcDCDBKBgNVHSAEQzBBMDUGDCsGAQQBsjEBAgEDCDAlMCMGCCsG
# AQUFBwIBFhdodHRwczovL3NlY3RpZ28uY29tL0NQUzAIBgZngQwBBAIwRAYDVR0f
# BD0wOzA5oDegNYYzaHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdvUlNBVGlt
# ZVN0YW1waW5nQ0EuY3JsMHQGCCsGAQUFBwEBBGgwZjA/BggrBgEFBQcwAoYzaHR0
# cDovL2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUlNBVGltZVN0YW1waW5nQ0EuY3J0
# MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG9w0B
# AQwFAAOCAgEAc9rtaHLLwrlAoTG7tAOjLRR7JOe0WxV9qOn9rdGSDXw9NqBp2fOa
# MNqsadZ0VyQ/fg882fXDeSVsJuiNaJPO8XeJOX+oBAXaNMMU6p8IVKv/xH6WbCvT
# lOu0bOBFTSyy9zs7WrXB+9eJdW2YcnL29wco89Oy0OsZvhUseO/NRaAA5PgEdrtX
# xZC+d1SQdJ4LT03EqhOPl68BNSvLmxF46fL5iQQ8TuOCEmLrtEQMdUHCDzS4iJ3I
# IvETatsYL254rcQFtOiECJMH+X2D/miYNOR35bHOjJRs2wNtKAVHfpsu8GT726QD
# MRB8Gvs8GYDRC3C5VV9HvjlkzrfaI1Qy40ayMtjSKYbJFV2Ala8C+7TRLp04fDXg
# DxztG0dInCJqVYLZ8roIZQPl8SnzSIoJAUymefKithqZlOuXKOG+fRuhfO1WgKb0
# IjOQ5IRT/Cr6wKeXqOq1jXrO5OBLoTOrC3ag1WkWt45mv1/6H8Sof6ehSBSRDYL8
# vU2Z7cnmbDb+d0OZuGktfGEv7aOwSf5bvmkkkf+T/FdpkkvZBT9thnLTotDAZNI6
# QsEaA/vQ7ZohuD+vprJRVNVMxcofEo1XxjntXP/snyZ2rWRmZ+iqMODSrbd9sWpB
# J24DiqN04IoJgm6/4/a3vJ4LKRhogaGcP24WWUsUCQma5q6/YBXdhvUwggbsMIIE
# 1KADAgECAhAwD2+s3WaYdHypRjaneC25MA0GCSqGSIb3DQEBDAUAMIGIMQswCQYD
# VQQGEwJVUzETMBEGA1UECBMKTmV3IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5IENp
# dHkxHjAcBgNVBAoTFVRoZSBVU0VSVFJVU1QgTmV0d29yazEuMCwGA1UEAxMlVVNF
# UlRydXN0IFJTQSBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eTAeFw0xOTA1MDIwMDAw
# MDBaFw0zODAxMTgyMzU5NTlaMH0xCzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVh
# dGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3Rp
# Z28gTGltaXRlZDElMCMGA1UEAxMcU2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBD
# QTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAMgbAa/ZLH6ImX0BmD8g
# kL2cgCFUk7nPoD5T77NawHbWGgSlzkeDtevEzEk0y/NFZbn5p2QWJgn71TJSeS7J
# Y8ITm7aGPwEFkmZvIavVcRB5h/RGKs3EWsnb111JTXJWD9zJ41OYOioe/M5YSdO/
# 8zm7uaQjQqzQFcN/nqJc1zjxFrJw06PE37PFcqwuCnf8DZRSt/wflXMkPQEovA8N
# T7ORAY5unSd1VdEXOzQhe5cBlK9/gM/REQpXhMl/VuC9RpyCvpSdv7QgsGB+uE31
# DT/b0OqFjIpWcdEtlEzIjDzTFKKcvSb/01Mgx2Bpm1gKVPQF5/0xrPnIhRfHuCkZ
# pCkvRuPd25Ffnz82Pg4wZytGtzWvlr7aTGDMqLufDRTUGMQwmHSCIc9iVrUhcxIe
# /arKCFiHd6QV6xlV/9A5VC0m7kUaOm/N14Tw1/AoxU9kgwLU++Le8bwCKPRt2ieK
# BtKWh97oaw7wW33pdmmTIBxKlyx3GSuTlZicl57rjsF4VsZEJd8GEpoGLZ8DXv2D
# olNnyrH6jaFkyYiSWcuoRsDJ8qb/fVfbEnb6ikEk1Bv8cqUUotStQxykSYtBORQD
# Hin6G6UirqXDTYLQjdprt9v3GEBXc/Bxo/tKfUU2wfeNgvq5yQ1TgH36tjlYMu9v
# GFCJ10+dM70atZ2h3pVBeqeDAgMBAAGjggFaMIIBVjAfBgNVHSMEGDAWgBRTeb9a
# qitKz1SA4dibwJ3ysgNmyzAdBgNVHQ4EFgQUGqH4YRkgD8NBd0UojtE1XwYSBFUw
# DgYDVR0PAQH/BAQDAgGGMBIGA1UdEwEB/wQIMAYBAf8CAQAwEwYDVR0lBAwwCgYI
# KwYBBQUHAwgwEQYDVR0gBAowCDAGBgRVHSAAMFAGA1UdHwRJMEcwRaBDoEGGP2h0
# dHA6Ly9jcmwudXNlcnRydXN0LmNvbS9VU0VSVHJ1c3RSU0FDZXJ0aWZpY2F0aW9u
# QXV0aG9yaXR5LmNybDB2BggrBgEFBQcBAQRqMGgwPwYIKwYBBQUHMAKGM2h0dHA6
# Ly9jcnQudXNlcnRydXN0LmNvbS9VU0VSVHJ1c3RSU0FBZGRUcnVzdENBLmNydDAl
# BggrBgEFBQcwAYYZaHR0cDovL29jc3AudXNlcnRydXN0LmNvbTANBgkqhkiG9w0B
# AQwFAAOCAgEAbVSBpTNdFuG1U4GRdd8DejILLSWEEbKw2yp9KgX1vDsn9FqguUlZ
# kClsYcu1UNviffmfAO9Aw63T4uRW+VhBz/FC5RB9/7B0H4/GXAn5M17qoBwmWFzz
# tBEP1dXD4rzVWHi/SHbhRGdtj7BDEA+N5Pk4Yr8TAcWFo0zFzLJTMJWk1vSWVgi4
# zVx/AZa+clJqO0I3fBZ4OZOTlJux3LJtQW1nzclvkD1/RXLBGyPWwlWEZuSzxWYG
# 9vPWS16toytCiiGS/qhvWiVwYoFzY16gu9jc10rTPa+DBjgSHSSHLeT8AtY+dwS8
# BDa153fLnC6NIxi5o8JHHfBd1qFzVwVomqfJN2Udvuq82EKDQwWli6YJ/9GhlKZO
# qj0J9QVst9JkWtgqIsJLnfE5XkzeSD2bNJaaCV+O/fexUpHOP4n2HKG1qXUfcb9b
# Q11lPVCBbqvw0NP8srMftpmWJvQ8eYtcZMzN7iea5aDADHKHwW5NWtMe6vBE5jJv
# HOsXTpTDeGUgOw9Bqh/poUGd/rG4oGUqNODeqPk85sEwu8CgYyz8XBYAqNDEf+oR
# nR4GxqZtMl20OAkrSQeq/eww2vGnL8+3/frQo4TZJ577AWZ3uVYQ4SBuxq6x+ba6
# yDVdM3aO8XwgDCp3rrWiAoa6Ke60WgCxjKvj+QrJVF3UuWp0nr1IrpgxggQtMIIE
# KQIBATCBkjB9MQswCQYDVQQGEwJHQjEbMBkGA1UECBMSR3JlYXRlciBNYW5jaGVz
# dGVyMRAwDgYDVQQHEwdTYWxmb3JkMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQx
# JTAjBgNVBAMTHFNlY3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcgQ0ECEQCQOX+a0ko6
# E/K9kV8IOKlDMA0GCWCGSAFlAwQCAgUAoIIBazAaBgkqhkiG9w0BCQMxDQYLKoZI
# hvcNAQkQAQQwHAYJKoZIhvcNAQkFMQ8XDTIyMTAwNTE3NDU1NVowPwYJKoZIhvcN
# AQkEMTIEMKiudVf5XFpBYf1mWGCoR3m1+0uxdpID7aQNkdVxk8aN6rlkHIJUk6oz
# aJ8vf2gyITCB7QYLKoZIhvcNAQkQAgwxgd0wgdowgdcwFgQUqzQBOqxAlzGfCBrw
# sxjhg/gPeIEwgbwEFALWW5Xig3DBVwCV+oj5I92Tf62PMIGjMIGOpIGLMIGIMQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKTmV3IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5
# IENpdHkxHjAcBgNVBAoTFVRoZSBVU0VSVFJVU1QgTmV0d29yazEuMCwGA1UEAxMl
# VVNFUlRydXN0IFJTQSBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eQIQMA9vrN1mmHR8
# qUY2p3gtuTANBgkqhkiG9w0BAQEFAASCAgAFjt7blh3KpyQkC978eZQZLe9V3Fb2
# pHHklK1LuYHqpqDgz6FOBTbO8dIbnPPPfQyaK6LW1Hlo3inJG8jwYzuD7lB0XQPQ
# 6z8N9/TEYSCZNTTyxWjuTtY8R8r1LxiIMwjZA10MDC9RqgumpZaRyWbFfBdWDYw9
# qSFHbnivdY0aFmyVLlhFVSp8RQgDtqvnI8YGeKS4c01jJH7OszdUPPO1fLD+lXy3
# DSSkdmyQmbqum/jQL70FTS196yDV5HnJ2yctrDn2z4Xr0JKaoSF1LV27OkeavEe1
# xwCgDAzCNCXgLgE+3fLTkfW14/5o4qXGlwQc/6XFqqgs/k3DnPsujmXGgAtLHlnv
# /pvsOrVUR+Xz7IsACqJKy5VeayCM9lhrTw/mZQcAkOjI1TI2ESOGSjPzZ6zHXGQ6
# yiUlw7KmKA0K69bghAmu1Drk3doZaSFjVXsuw6fg8nZtB1Wv59yLH4YDUXlZSqpo
# ESlFac+oiVj6q0obpDOFswJ3sXJ5xZGjBV+w/t+7dWx47Z2WoQeJopcsYEIS4Bgr
# ao3XeV3pfhDOuWs0svNDDFNxzIo7O1CKmFsE15QLOxEUzTat0zMzhMibJp9Gw14w
# ci0J6IVueWkeK2R10EpQuKQc86PZ7MGsZo9E2D39Xh5GPoqOpVif9tnA/ObI8SNW
# XjV+r7r91pfWww==
# SIG # End signature block
