/******************************************************************************

   Copyright (c) Hilscher GmbH. All Rights Reserved.

 ******************************************************************************

  Last Modification:
    @version $Id:$

  Description:
    servX HTTP Server
    Standard configuration for hilscher products

  Changes:
    Date        Author        Description
  ---------------------------------------------------------------------------
    2011-09-28  NC            - initial version

*******************************************************************************/

/******************************************************************************/
/*  Includes                                                                  */
/******************************************************************************/
#ifdef __RCX__
#include "TLR_Includes.h"
#include "rX_Includes.h"
#endif

#include "SxConfig.h"
#include <stdio.h>

/******************************************************************************/
/*  servX Module Configuration Structures                                     */
/******************************************************************************/

//-----------------------------------
// modTranslate: URI Translation
//-----------------------------------
static SX_TRANS_ENTRY_T s_atTransTable[] = {
  {"", "index.htm"},
};

/** module configuration */
static SX_TRANSLATE_CONFIG_T s_tCfgModTranslate = {
  NULL,
  s_atTransTable,
  MAX_CNT(s_atTransTable)
};

//-----------------------------------
// modDigestAuth: Protection against replay attacks
//-----------------------------------
static SX_DIGEST_AUTHORIZATION_T s_atAuthorizations[] = {
  {"^/upload",    NULL,             "upload"},
  {"^/fwupdate",  NULL,             "firmware"},
  {"^/reset",     NULL,             "reset"},
};

/** module configuration */
static SX_DIGESTAUTH_CONFIG_T s_tCfgModDigestAuth = {
  s_atAuthorizations,
  MAX_CNT(s_atAuthorizations),
  "PORT_1/SECURITY.CFG",
};

//-----------------------------------
// modTypemap: Content Type Recognition (mime)
//-----------------------------------
static SX_TYPEMAP_ENTRY_T s_atTypeMap[] = {
  /* text media types */
  {"htm stm sht html", "text/html; charset=utf-8"},
  {"css", "text/css"},
  {"xml", "text/xml"},
  {"csv", "text/csv"},
  {"txt", "text/plain"},
  {"rtf rtx", "text/richtext"},
  /* image media types */
  {"gif", "image/gif"},
  {"bmp", "image/bmp"},
  {"jpg jpeg", "image/jpeg"},
  {"png", "image/png"},
  {"svg", "image/svg+xml"},
  {"ief", "image/ief"},
  {"cgm", "image/cgm"},
  {"tif tiff", "image/tiff"},
  /* application media types */
  {"js", "application/x-javascript"},
  {"jsn json", "application/json"},
  {"ps", "application/postscript"},
  /* user defined types */
};

/** module configuration */
static SX_TYPEMAP_CONFIG_T s_tCfgModTypemap = {
  s_atTypeMap,
  MAX_CNT(s_atTypeMap),
  NULL,
};

//-----------------------------------
// modFilesystem: File System Access
//-----------------------------------
/** module configuration */
static SX_FS_CONFIG_T s_tCfgModFilesystem = {
  "error404.htm",
  1,
  SX_DOCUMENT_ROOT,
};

//-----------------------------------
// modUpload: File Upload via HTTP
//-----------------------------------
/** module configuration */
static SX_UP_CONFIG_T s_tCfgModUpload = {
  .pszRoot  = SX_SERVER_ROOT,
  .ulOptions = SX_UP_OPT_ALL,
  .pfnPreProc = NULL,
  .pvPreProcParam = NULL,
  .pfnPostProc = NULL,
  .pvPostProcParam = NULL,
  .pszTitle = NULL,
  .ulFileListSize = 0,
};

//-----------------------------------
// modSSI: Server Side Includes
//-----------------------------------
/** SSI function*/
/** Get hardware information */
void ssi_getHwInfo(SX_ENV_PT ptEnv, SX_DOC_PT ptDoc, const char* pszArg, void* pvCtx) {
  AT_CM_IDENTIFICATION_T 	tIdentData;
  unsigned char 					abMac[6];
  char 										szBuffer[64];

  if(TLR_S_OK == Drv_SecGetIdentificationData( &tIdentData))
  {
    sprintf(szBuffer,"%lu",tIdentData.ulDeviceNumber);
    sx_setenv(ptEnv,"dev.DEVNUM",szBuffer,1);
    sprintf(szBuffer,"%lu",tIdentData.ulSerialNumber);
    sx_setenv(ptEnv,"dev.SERIAL",szBuffer,1);
    sprintf(szBuffer,"%u",tIdentData.usDeviceClass);
    sx_setenv(ptEnv,"dev.CLASS",szBuffer,1);
  }

  if( TLR_S_OK == Drv_SecGetMacAddress(abMac,0) )
  {
    sprintf(szBuffer,"%02x:%02x:%02x:%02x:%02x:%02x",
                     abMac[0],abMac[1],abMac[2],abMac[3],abMac[4],abMac[5]);
    sx_setenv(ptEnv,"dev.HWADDR",szBuffer,1);
  }
}

/** list of SSI callback function */
static SX_SSI_FUNC_T s_atSsiFunctions[] = {
  {"dev.getHwInfo",  ssi_getHwInfo, NULL},\
  {"", NULL, NULL},
};

/** module configuration */
static SX_SSI_CONFIG_T s_tCfgModSsi = {
  MAX_CNT(s_atSsiFunctions),
  s_atSsiFunctions,
  SX_DOCUMENT_ROOT,
  NULL,
  NULL,
  NULL,
};

//-----------------------------------
// modNxReset: Nx Reset Module
//-----------------------------------
/** module configuration */
static SX_NXRESET_CONFIG_T s_tCfgModNxReset = {
    NULL
};

//-----------------------------------
// modNxFwUpdate: Nx Firmware Module
//-----------------------------------
/** module configuration */
static SX_NXFW_CONFIG_T s_tCfgModNxFwUpdate = {
    "PORT_0",
    NULL,
    "Reset device to activate new firmware.",
    (SX_NXFW_UPD_STORE_SAFETY |
     SX_NXFW_UPD_CHECK_PROTOCOL_CLASS |
     SX_NXFW_UPD_CHECK_COMMUNICATION_CLASS)
};


/******************************************************************************/
/*  servX Module Pool Configuration                                           */
/******************************************************************************/
static SX_MOD_REFERENCE_T s_atModules[] = {
  {SX_MOD_TRANSLATE,  &s_tCfgModTranslate},
  {SX_MOD_DIGESTAUTH, &s_tCfgModDigestAuth},
  {SX_MOD_TYPEMAP,    &s_tCfgModTypemap},
  {SX_MOD_FILESYSTEM, &s_tCfgModFilesystem},
  {SX_MOD_UPLOAD,     &s_tCfgModUpload},
  {SX_MOD_SSI,        &s_tCfgModSsi},
  {SX_MOD_NXRESET,    &s_tCfgModNxReset},
  {SX_MOD_NXFIRMWARE, &s_tCfgModNxFwUpdate},
};

typedef enum {
  MOD_IDX_TRANSLATE = 0,
  MOD_IDX_DIGESTAUTH,
  MOD_IDX_TYPES,
  MOD_IDX_FILESYSTEM,
  MOD_IDX_UPLOAD,
  MOD_IDX_SSI,
  MOD_IDX_RESET,
  MOD_IDX_FWUPDATE,
} MODULE_INDICES;


/******************************************************************************/
/*  servX Module Chain Configuration                                           */
/******************************************************************************/
static unsigned long s_aulStdModChain[] = {
  MOD_IDX_TRANSLATE,
  MOD_IDX_DIGESTAUTH,
  MOD_IDX_TYPES,
  SX_MOD_IDX_NONE,
  MOD_IDX_FILESYSTEM,
  SX_MOD_IDX_NONE,
};

typedef enum {
  PHASE_TRANSLATE = 0,
  PHASE_AUTH,
  PHASE_TYPES,
  PHASE_INPUT_HANDLER,
  PHASE_GENERATE,
  PHASE_OUTPUT_FILTER
}PHASES;


/******************************************************************************/
/*  servX Module Pool Rule Set                                                */
/******************************************************************************/
static SX_MODPOOL_RULE_T s_atRuleSet[] =
/*if<CONDITION> match <ARGUMENT> enable <MODULE> at           <PHASE> */
{
  { SX_COND_PATH,     "^/upload$",      MOD_IDX_UPLOAD,       PHASE_GENERATE },
  { SX_COND_PATH,     "^/reset$",       MOD_IDX_RESET,        PHASE_GENERATE },
  { SX_COND_PATH,     "^/fwupdate$",    MOD_IDX_FWUPDATE,     PHASE_GENERATE },
  { SX_COND_SUFFIX,   "sht",            MOD_IDX_SSI,          PHASE_OUTPUT_FILTER},
  { SX_COND_SUFFIX,   "stm",            MOD_IDX_SSI,          PHASE_OUTPUT_FILTER},
  { SX_COND_SUFFIX,   "jsn",            MOD_IDX_SSI,          PHASE_OUTPUT_FILTER},
};


/******************************************************************************/
/* Main Config Structure (declared global for use in rcX config file)         */
/******************************************************************************/
SX_HTTP_STARTUPPARAMETER_T g_tSxConfig = {
    /** Header */
    .ulTaskIdentifier =  TLR_TASK_SERVX,
    .ulParamVersion   = SX_HTTP_STARTUP_PARAM_VERSION_3,
    /** Module list */
    .patModules   = s_atModules,
    .ulNumModules = MAX_CNT(s_atModules),
    /** Standard module chain */
    .paulStdChainModules  = s_aulStdModChain,
    .ulNumChainPhases     = MAX_CNT(s_aulStdModChain),
    /** Exception Rules */
    .patRuleSet = s_atRuleSet,
    .ulNumRules = MAX_CNT(s_atRuleSet),
    /** Buffer settings */
    .ulBufferSize      = SX_BUFFER_SIZE,
    .pabBufferLocation = NULL,
    /** Connection handling */
    .ulNumConnections     = SX_CONNECTION,
    .ulKeepAlive          = SX_HTTP_KEEP_ALIVE_DEF,
    .ulPort               = SX_HTTP_PORT_DEF,
    .ulMaxRequestBodySize = SX_MAX_REQUEST_BODY,
    /** Server settings */
    .pszServer            = SX_HTTP_SERVERNAME_DEF,
    .ptTcpConfig          = SX_HTTP_TCP_CONFIG_DEF,
    .pfnCreateVar         = SX_HTTP_CREATE_VAR_DEF,
    .pvCreateVarConfig    = SX_HTTP_CREATE_VAR_CFG_DEF,
    .pfnErrCallback       = SX_HTTP_ERR_CBK_DEF,
    .pvErrCallbackParam   = SX_HTTP_ERR_CBK_CFG_DEF,
    .pfnInitCallback      = SX_HTTP_INIT_CBK_DEF,
    .pvInitCallbackParam  = SX_HTTP_INIT_CBK_CFG_DEF,
};


