/*******************************************************************************
*
* Copyright (c) 2015-2019 Intel Corporation.  All rights reserved.
*
* This software is available to you under a choice of one of two
* licenses.  You may choose to be licensed under the terms of the GNU
* General Public License (GPL) Version 2, available from the file
* COPYING in the main directory of this source tree, or the
* OpenFabrics.org BSD license below:
*
*   Redistribution and use in source and binary forms, with or
*   without modification, are permitted provided that the following
*   conditions are met:
*
*    - Redistributions of source code must retain the above
*	copyright notice, this list of conditions and the following
*	disclaimer.
*
*    - Redistributions in binary form must reproduce the above
*	copyright notice, this list of conditions and the following
*	disclaimer in the documentation and/or other materials
*	provided with the distribution.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
* EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
* BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
* ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
* CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
* SOFTWARE.
*
*******************************************************************************/
/*$FreeBSD$*/

#ifndef _IW_IXL_H_
#define _IW_IXL_H_

#include <sys/param.h>
#include <rdma/ib_verbs.h>
#include <rdma/iw_cm.h>

#include "i40iw_status.h"
#include "i40iw_osdep.h"
#include "i40iw_d.h"
#include "i40iw_hmc.h"
#include "i40iw_type.h"
#include "i40iw_p.h"
#include "i40iw_user.h"
#include "i40iw_puda.h"
#include "iw_ixl_ucontext.h"
#include "iw_ixl_pble.h"
#include "iw_ixl_verbs.h"
#include "iw_ixl_cm.h"
#include "iw_ixl_dbg.h"

#include "ixl_iw.h"

#if defined(INET6) && defined(IW_IXL_RDMA_4_9)
#define IPV6_SUPPORT
#else
#undef IPV6_SUPPORT
#endif

#define I40IW_FW_VERSION  2

#define I40IW_ARP_ADD     1
#define I40IW_ARP_DELETE  2
#define I40IW_ARP_RESOLVE 3

#define IW_CCQ_SIZE         (I40IW_CQP_SW_SQSIZE_2048 + 1)

#define IW_FIRST_QPN 1

#define I40IW_EVENT_TIMEOUT		100000

#define	I40IW_NO_VLAN			0xffff
#define I40IW_NO_QSET			0xffff

#define IW_HMC_OBJ_TYPE_NUM ARRAY_SIZE(iw_hmc_obj_types)

#define I40IW_MAX_PAGES_PER_FMR		512
#define I40IW_MIN_PAGES_PER_FMR           1

#define IW_CFG_FPM_QP_COUNT 32768

#ifndef VLAN_ETH_HLEN
#define VLAN_ETH_HLEN 18
#endif

#if __FreeBSD_version >= 1200072
#define I40IW_TAILQ_FOREACH CK_STAILQ_FOREACH
#define I40IW_TAILQ_FOREACH_SAFE CK_STAILQ_FOREACH_SAFE
#else
#define I40IW_TAILQ_FOREACH TAILQ_FOREACH
#define I40IW_TAILQ_FOREACH_SAFE TAILQ_FOREACH_SAFE
#endif

#ifdef IW_IXL_RDMA_4_9
#define I40IW_CQP_COMPL_RQ_WQE_FLUSHED    2
#define I40IW_CQP_COMPL_SQ_WQE_FLUSHED    3
#endif

struct i40iw_handler;

struct i40iw_cqp_compl_info {
	u32    op_ret_val;
	u16    maj_err_code;
	u16    min_err_code;
	bool   error;
	u8     op_code;
};

#define i40iw_pr_err(fmt, args ...) printf("%s: "fmt, __func__, ## args)

struct i40iw_cqp_request {
	struct cqp_commands_info info;
	wait_queue_head_t waitq;
	struct list_head list;
	atomic_t refcount;
	void   (*callback_fcn) (struct i40iw_cqp_request *, u32);
	void  *param;
	struct i40iw_cqp_compl_info compl_info;
	bool   waiting;
	bool   request_done;
	bool   dynamic;
};

struct i40iw_cqp {
	struct i40iw_sc_cqp sc_cqp;
	spinlock_t req_lock;	/* cqp request list */
	wait_queue_head_t waitq;
	struct i40iw_dma_mem sq;
	struct i40iw_dma_mem host_ctx;
	u64   *scratch_array;
	struct i40iw_cqp_request *cqp_requests;
	struct list_head cqp_avail_reqs;
	struct list_head cqp_pending_reqs;
};

struct i40iw_ccq {
	struct i40iw_sc_cq sc_cq;
	spinlock_t lock;	/* ccq control */
	wait_queue_head_t waitq;
	struct i40iw_dma_mem mem_cq;
	struct i40iw_dma_mem shadow_area;
};

struct i40iw_ceq {
	struct i40iw_sc_ceq sc_ceq;
	struct i40iw_dma_mem mem;
	u32    msix_idx;
	struct i40iw_device *iwdev;
	struct task irq_task;
	struct taskqueue *irq_tq;
};

struct i40iw_aeq {
	struct i40iw_sc_aeq sc_aeq;
	struct i40iw_dma_mem mem;
};

struct i40iw_arp_entry {
	u32    ip_addr[4];
	u8     mac_addr[ETH_ALEN];
};

enum init_completion_state {
	INVALID_STATE = 0,
	INITIAL_STATE,
	CQP_CREATED,
	RDMA_FEATURES_RETRIEVED,
	HMC_OBJS_CREATED,
	PBLE_CHUNK_MEM,
	CCQ_CREATED,
	AEQ_CREATED,
	CEQ_CREATED,
	ILQ_CREATED,
	IEQ_CREATED,
	IP_ADDR_REGISTERED,
	RDMA_DEV_REGISTERED
};

struct i40iw_msix_vector {
	u32    idx;
	struct resource *res;
	void  *tag;
	u32    cpu_affinity;
};

struct i40iw_dev_ctx {
	bus_space_tag_t mem_bus_space_tag;
	bus_space_handle_t mem_bus_space_handle;
	bus_size_t mem_bus_space_size;
	struct device *dev;
};

struct i40iw_device {
	struct i40iw_ib_device *iwibdev;
	struct ifnet *ifp;
	wait_queue_head_t vchnl_waitq;
	struct i40iw_sc_dev sc_dev;
	struct i40iw_sc_vsi vsi;	/* Added for FSL VSI SUPPORT */
	struct i40iw_handler *hdl;
	struct ixl_iw_pf *ldev;
	struct i40iw_hw hw;
	struct i40iw_dev_ctx dev_ctx;
	struct i40iw_cm_core cm_core;
	u8    *mem_resources;
	i40iw_bits_t allocated_qps;
	i40iw_bits_t allocated_cqs;
	i40iw_bits_t allocated_mrs;
	i40iw_bits_t allocated_pds;
	i40iw_bits_t allocated_arps;
	struct i40iw_qp **qp_table;
	struct ixl_iw_msix msix_info;
	bool   msix_shared;
	u32    msix_count;
	struct i40iw_msix_vector *iw_msixtbl;
	struct ixl_iw_msix_mapping msix_mapping;

	struct i40iw_hmc_pble_rsrc *pble_rsrc;
	struct i40iw_arp_entry *arp_table;
	struct i40iw_cqp cqp;
	struct i40iw_ccq ccq;
	u32    ceqs_count;
	struct i40iw_ceq *ceqlist;
	struct i40iw_aeq aeq;
	u32    arp_table_size;
	u32    next_arp_index;
	spinlock_t resource_lock;	/* hw resource access */
	spinlock_t qptable_lock;
	u32    device_cap_flags;
	u8     resource_profile;
	u8     max_rdma_vfs;
	u8     iw_status;
	bool   push_mode;	/* Initialized with tunable */
	struct mutex pbl_mutex;
	struct task irq_task;
	struct taskqueue *irq_tq;
	struct i40iw_dma_mem obj_mem;
	struct i40iw_dma_mem obj_next;
	u8    *hmc_info_mem;
	u32    sd_type;
	enum init_completion_state init_state;
	u16    mac_ip_table_idx;
	u32    max_mr;
	u32    max_qp;
	u32    max_cq;
	u32    max_pd;
	u32    next_qp;
	u32    next_cq;
	u32    next_pd;
	u32    max_cqe;
	u32    mr_stagmask;
	u32    mpa_version;
	bool   closing;
	bool   reset;
	u32    used_pds;
	u32    used_cqs;
	u32    used_mrs;
	u32    used_qps;
	wait_queue_head_t close_wq;
	atomic_t use_count;
};

struct i40iw_ib_device {
	struct ib_device ibdev;
	struct i40iw_device *iwdev;
};

struct i40iw_handler {
	struct list_head list;
	struct i40iw_device device;
	struct ixl_iw_pf ldev;
};

/**
 * i40iw_alloc_resource - allocate a resource
 * @iwdev: device pointer
 * @resource_array: resource bit array:
 * @max_resources: maximum resource number
 * @req_resources_num: Allocated resource number
 * @next: next free id
 **/
static inline int
i40iw_alloc_resource(struct i40iw_device *iwdev,
		     i40iw_bits_t resource_array,
		     u32 max_resources,
		     u32 *req_resource_num,
		     u32 *next)
{
	u32    resource_num;
	unsigned long flags;

	spin_lock_irqsave(&iwdev->resource_lock, flags);
	resource_num = find_next_zero_bit(resource_array, max_resources, *next);
	if (resource_num >= max_resources) {
		resource_num = find_first_zero_bit(resource_array, max_resources);
		if (resource_num >= max_resources) {
			spin_unlock_irqrestore(&iwdev->resource_lock, flags);
			return -EOVERFLOW;
		}
	}
	set_bit(resource_num, resource_array);
	*next = resource_num + 1;
	if (*next == max_resources)
		*next = 0;
	*req_resource_num = resource_num;
	spin_unlock_irqrestore(&iwdev->resource_lock, flags);

	return 0;
}

/**
 * i40iw_free_resource - free a resource
 * @iwdev: device pointer
 * @resource_array: resource array for the resource_num
 * @resource_num: resource number to free
 **/
static inline void
i40iw_free_resource(struct i40iw_device *iwdev,
		    i40iw_bits_t resource_array,
		    u32 resource_num)
{
	unsigned long flags;

	spin_lock_irqsave(&iwdev->resource_lock, flags);
	clear_bit(resource_num, resource_array);
	spin_unlock_irqrestore(&iwdev->resource_lock, flags);
}

/**
 * to_iwdev - get device
 * @ibdev: ib device
 **/
static inline struct i40iw_device *
to_iwdev(struct ib_device *ibdev)
{
	return container_of(ibdev, struct i40iw_ib_device, ibdev)->iwdev;
}

/**
 * to_ucontext - get user context
 * @ibucontext: ib user context
 **/
static inline struct i40iw_ucontext *
to_ucontext(struct ib_ucontext *ibucontext)
{
	return container_of(ibucontext, struct i40iw_ucontext, ibucontext);
}

/**
 * to_iwpd - get protection domain
 * @ibpd: ib pd
 **/
static inline struct i40iw_pd *
to_iwpd(struct ib_pd *ibpd)
{
	return container_of(ibpd, struct i40iw_pd, ibpd);
}

/**
 * to_iwmr - get device memory region
 * @ibdev: ib memory region
 **/
static inline struct i40iw_mr *
to_iwmr(struct ib_mr *ibmr)
{
	return container_of(ibmr, struct i40iw_mr, ibmr);
}

/**
 * to_iwcq - get completion queue
 * @ibcq: ib cqdevice
 **/
static inline struct i40iw_cq *
to_iwcq(struct ib_cq *ibcq)
{
	return container_of(ibcq, struct i40iw_cq, ibcq);
}

/**
 * to_iwqp - get device qp
 * @ibqp: ib qp
 **/
static inline struct i40iw_qp *
to_iwqp(struct ib_qp *ibqp)
{
	return container_of(ibqp, struct i40iw_qp, ibqp);
}

#ifndef IW_IXL_RDMA_4_9
/**
 * to_i40iw_fr_page_list - get fast memreg page list
 * @ibfpl: ib fast memreg page list
 **/
static inline struct i40iw_fast_reg_page_list *
to_i40iw_fr_page_list(struct ib_fast_reg_page_list *ibfpl)
{
	return container_of(ibfpl, struct i40iw_fast_reg_page_list, ibfpl);
}

#endif
void   i40iw_flush_wqes(struct i40iw_device *, struct i40iw_qp *);
void   i40iw_manage_arp_cache(struct i40iw_device *, unsigned char *, u32 *, bool, u32);
int    i40iw_manage_apbvt(struct i40iw_device *, u16, bool);
void   i40iw_put_cqp_request(struct i40iw_cqp *, struct i40iw_cqp_request *);
struct i40iw_cqp_request *i40iw_get_cqp_request(struct i40iw_cqp *, bool);
void   i40iw_free_cqp_request(struct i40iw_cqp *, struct i40iw_cqp_request *);
u32    i40iw_initialize_hw_resources(struct i40iw_device *);
int    i40iw_register_rdma_device(struct i40iw_device *);
void   i40iw_port_ibevent(struct i40iw_device *);
enum i40iw_status_code i40iw_handle_cqp_op(struct i40iw_device *, struct i40iw_cqp_request *);
enum i40iw_status_code i40iw_manage_qhash(struct i40iw_device *, struct i40iw_cm_info *, enum i40iw_quad_entry_type, enum i40iw_quad_hash_manage_type, void *, bool);
void   i40iw_receive_ilq(struct i40iw_sc_vsi *, struct i40iw_puda_buf *);
void   i40iw_free_sqbuf(struct i40iw_sc_vsi *, void *);
enum i40iw_status_code i40iw_obj_aligned_mem(struct i40iw_device *, struct i40iw_dma_mem *, u32, u32);
void   i40iw_request_reset(struct i40iw_device *);
void   i40iw_destroy_rdma_device(struct i40iw_ib_device *);
void   i40iw_setup_cm_core(struct i40iw_device *);
int    i40iw_modify_qp(struct ib_qp *, struct ib_qp_attr *, int, struct ib_udata *);
void   i40iw_hw_modify_qp(struct i40iw_device *, struct i40iw_qp *, struct i40iw_modify_qp_info *, bool);
void   i40iw_cq_wq_destroy(struct i40iw_device *, struct i40iw_sc_cq *);
void   i40iw_next_iw_state(struct i40iw_qp *, u8, u8, u8, u8);
void   i40iw_cm_disconn(struct i40iw_qp *);
void   i40iw_free_qp_resources(struct i40iw_device *, struct i40iw_qp *, u32);
void   i40iw_cleanup_cm_core(struct i40iw_cm_core *);
void   i40iw_process_ceq(struct i40iw_device *, struct i40iw_ceq *);
void   i40iw_process_aeq(struct i40iw_device *);
int    i40iw_send_syn(struct i40iw_cm_node *, u32);
int    i40iw_send_reset(struct i40iw_cm_node *cm_node);
struct i40iw_cm_node *i40iw_find_node(struct i40iw_cm_core *, u16, u32 *, u16, u32 *, bool, bool);
enum i40iw_status_code i40iw_hw_flush_wqes(struct i40iw_device *, struct i40iw_sc_qp *, struct i40iw_qp_flush_info *, bool);
void   i40iw_gen_ae(struct i40iw_device *, struct i40iw_sc_qp *, struct i40iw_gen_ae_info *, bool);
#ifdef IW_IXL_RDMA_4_9
struct ib_mr *i40iw_reg_phys_mr(struct ib_pd *, u64, u64, int, u64 *);
#else
struct ib_mr *i40iw_reg_phys_mr(struct ib_pd *, struct ib_phys_buf *, int, int, u64 *);
#endif
void   i40iw_cleanup_pending_cqp_op(struct i40iw_device *);
void   i40iw_rem_pdusecount(struct i40iw_pd *, struct i40iw_device *);
void   i40iw_add_pdusecount(struct i40iw_pd *);
void   i40iw_rem_devusecount(struct i40iw_device *);
void   i40iw_add_devusecount(struct i40iw_device *);
void   i40iw_copy_ip_ntohl(u32 *, __be32 *);
void   i40iw_add_ref(struct ib_qp *);
void   i40iw_rem_ref(struct ib_qp *);
struct ib_qp *i40iw_get_qp(struct ib_device *, int);
void   i40iw_cq_ceq_mapping_watch(struct i40iw_device *, unsigned int, bool);

#endif				/* _IW_IXL_H_ */
