/**************************************************************************

Copyright (c) 2016 - 2021, Intel Corporation

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice,
      this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of Intel Corporation nor the names of its contributors
      may be used to endorse or promote products derived from this software
      without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

***************************************************************************/

#ifndef _ICE_TYPE_H_
#define _ICE_TYPE_H_

#ifndef ETH_ALEN
#define ETH_ALEN	6
#endif /* ETH_ALEN */

#ifndef ETH_HEADER_LEN
#define ETH_HEADER_LEN	14
#endif /* ETH_HEADER_LEN */

#ifndef BIT
#define BIT(a) (1UL << (a))
#endif /* BIT */
#ifndef BIT_ULL
#define BIT_ULL(a) (1ULL << (a))
#endif /* BIT_ULL */

#ifndef BITS_PER_BYTE
#define BITS_PER_BYTE	8
#endif

#ifndef _FORCE_
#define _FORCE_
#endif

#define ICE_BYTES_PER_WORD	2
#define ICE_BYTES_PER_DWORD	4
#define ICE_MAX_TRAFFIC_CLASS	8
#ifdef ADQ_SUPPORT
#define ICE_CHNL_MAX_TC		16
#endif /* ADQ_SUPPORT */

#ifndef DIVIDE_AND_ROUND_UP
#define DIVIDE_AND_ROUND_UP(a, b) (((a) + (b) - 1) / (b))
#endif

#ifndef ROUND_UP
/**
 * ROUND_UP - round up to next arbitrary multiple (not a power of 2)
 * @a: value to round up
 * @b: arbitrary multiple
 *
 * Round up to the next multiple of the arbitrary b.
 * Note, when b is a power of 2 use ICE_ALIGN() instead.
 */
#define ROUND_UP(a, b)	((b) * DIVIDE_AND_ROUND_UP((a), (b)))
#endif

#ifndef MIN_T
#define MIN_T(_t, _a, _b)	min((_t)(_a), (_t)(_b))
#endif

#ifndef IS_ASCII
#define IS_ASCII(_ch)	((_ch) < 0x80)
#endif

#ifdef C99
#ifdef STRUCT_HACK_VAR_LEN
#undef STRUCT_HACK_VAR_LEN
#endif /* STRUCT_HACK_VAR_LEN */
#define STRUCT_HACK_VAR_LEN
#ifndef ice_struct_size
/**
 * ice_struct_size - size of struct with C99 flexible array member
 * @ptr: pointer to structure
 * @field: flexible array member (last member of the structure)
 * @num: number of elements of that flexible array member
 */
#define ice_struct_size(ptr, field, num) \
	(sizeof(*(ptr)) + sizeof(*(ptr)->field) * (num))
#endif /* ice_struct_size */
#else /* !C99 */
#ifndef STRUCT_HACK_VAR_LEN
#define STRUCT_HACK_VAR_LEN     1
#elif !((STRUCT_HACK_VAR_LEN == 0) || (STRUCT_HACK_VAR_LEN == 1))
#error STRUCT_HACK_VAR_LEN must be 0 or 1
#endif /* STRUCT_HACK_VAR_LEN */
#ifndef ice_struct_size
/**
 * ice_struct_size - size of struct with variable-length object as last member
 * @ptr: pointer to structure
 * @field: variable-length object (last member of the structure)
 * @num: number of elements of that variable-length object (array)
 */
#define ice_struct_size(ptr, field, num) \
	(sizeof(*(ptr)) + sizeof(*(ptr)->field) * ((num) - STRUCT_HACK_VAR_LEN))
#endif /* ice_struct_size */
#endif /* !C99 */

#ifndef FLEX_ARRAY_SIZE
#define FLEX_ARRAY_SIZE(_ptr, _mem, cnt) ((cnt) * sizeof(_ptr->_mem[0]))
#endif /* FLEX_ARRAY_SIZE */

#include "ice_status.h"
#ifdef E830_SUPPORT
#define GL_MDCK_TX_TDPU				0x00049348
#define GL_MDCK_TX_TDPU_RCU_ANTISPOOF_ITR_DIS_M	BIT(1)
#define PRTMAC_HSEC_CTL_TX_PAUSE_QUANTA_MAX_INDEX 8
#define PRTMAC_HSEC_CTL_TX_PAUSE_QUANTA(_i) \
			((_i) == PRTMAC_HSEC_CTL_TX_PAUSE_QUANTA_MAX_INDEX ? \
			 PRTMAC_CL01_PAUSE_QUANTA : 0x0)
#define PRTMAC_HSEC_CTL_TX_PAUSE_QUANTA_HSEC_CTL_TX_PAUSE_QUANTA_M	\
				PRTMAC_CL01_PAUSE_QUANTA_CL0_PAUSE_QUANTA_M
#define PRTMAC_HSEC_CTL_TX_PAUSE_REFRESH_TIMER(_i) \
			((_i) == PRTMAC_HSEC_CTL_TX_PAUSE_QUANTA_MAX_INDEX ? \
			 PRTMAC_CL01_QUANTA_THRESH : 0x0)
#define PRTMAC_HSEC_CTL_TX_PAUSE_REFRESH_TIMER_M	\
				PRTMAC_CL01_QUANTA_THRESH_CL0_QUANTA_THRESH_M
#define GLNVM_ULD_PCIER_DONE_M		BIT(0)
#define GLNVM_ULD_PCIER_DONE_1_M	BIT(1)
#define GLNVM_ULD_PCIER_DONE_2_M	BIT(9)
#define GLGEN_RTRIG			0x000B8180 /* Reset Source: CORER */
#define GLGEN_RTRIG_CORER_M		BIT(0)
#define GLGEN_RTRIG_GLOBR_M		BIT(1)

#include "ice_hw_autogen_cnv.h"
#else /* !E830_SUPPORT */
#include "ice_hw_autogen.h"
#endif /* !E830_SUPPORT */
#ifndef QV_SUPPORT
#include "ice_devids.h"
#endif /* QV_SUPPORT */
#ifdef SDK_SUPPORT
#include "ice_alloc.h" /* must be included before osdep.h */
#endif /* SDK_SUPPORT */
#include "ice_osdep.h"
#ifdef QV_SUPPORT
#include "ice_devids.h"
#endif /* QV_SUPPORT */
#if !defined(LINUX_SUPPORT)
#include "ice_bitops.h" /* Must come before ice_controlq.h */
#endif /* !LINUX_SUPPORT */
#include "ice_controlq.h"
#include "ice_lan_tx_rx.h"
#include "ice_flex_type.h"
#include "ice_protocol_type.h"
#ifndef NO_SBQ_SUPPORT
#include "ice_sbq_cmd.h"
#endif /* !NO_SBQ_SUPPORT */
#include "ice_vlan_mode.h"
#ifdef FWLOG_SUPPORT_V2
#include "ice_fwlog.h"
#endif /* FWLOG_SUPPORT_V2 */

#ifndef ice_is_pow2
/**
 * ice_is_pow2 - check if integer value is a power of 2
 * @val: unsigned integer to be validated
 */
static inline bool ice_is_pow2(u64 val)
{
	return (val && !(val & (val - 1)));
}
#endif /* ice_is_pow2 */

#ifndef ice_ilog2
/**
 * ice_ilog2 - Calculates integer log base 2 of a number
 * @n: number on which to perform operation
 */
static inline int ice_ilog2(u64 n)
{
	int i;

	for (i = 63; i >= 0; i--)
		if (((u64)1 << i) & n)
			return i;

	return -1;
}
#endif /* ice_ilog2 */

static inline bool ice_is_tc_ena(ice_bitmap_t bitmap, u8 tc)
{
#if defined(LINUX_SUPPORT) || defined(DPDK_SUPPORT)
	return ice_is_bit_set(&bitmap, tc);
#else
	return !!(bitmap & BIT(tc));
#endif /* LINUX_SUPPORT || DPDK_SUPPORT */
}

#ifndef DIV_64BIT
#ifndef LINUX_SUPPORT
#define DIV_64BIT(n, d) ((n) / (d))
#else
#define DIV_64BIT(n, d) div64_long((n), (d))
#endif /* LINUX_SUPPORT */
#endif /* DIV_64BIT */

static inline u64 round_up_64bit(u64 a, u32 b)
{
	return DIV_64BIT(((a) + (b) / 2), (b));
}

static inline u32 ice_round_to_num(u32 N, u32 R)
{
	return ((((N) % (R)) < ((R) / 2)) ? (((N) / (R)) * (R)) :
		((((N) + (R) - 1) / (R)) * (R)));
}

/* Driver always calls main vsi_handle first */
#define ICE_MAIN_VSI_HANDLE		0

/* Switch from ms to the 1usec global time (this is the GTIME resolution) */
#define ICE_MS_TO_GTIME(time)		((time) * 1000)

/* Data type manipulation macros. */
#define ICE_HI_DWORD(x)		((u32)((((x) >> 16) >> 16) & 0xFFFFFFFF))
#define ICE_LO_DWORD(x)		((u32)((x) & 0xFFFFFFFF))
#define ICE_HI_WORD(x)		((u16)(((x) >> 16) & 0xFFFF))
#define ICE_LO_WORD(x)		((u16)((x) & 0xFFFF))

/* debug masks - set these bits in hw->debug_mask to control output */
#ifndef LINUX_SUPPORT
#define ICE_DBG_TRACE		BIT_ULL(0) /* for function-trace only */
#endif
#define ICE_DBG_INIT		BIT_ULL(1)
#define ICE_DBG_RELEASE		BIT_ULL(2)
#define ICE_DBG_FW_LOG		BIT_ULL(3)
#define ICE_DBG_LINK		BIT_ULL(4)
#define ICE_DBG_PHY		BIT_ULL(5)
#define ICE_DBG_QCTX		BIT_ULL(6)
#define ICE_DBG_NVM		BIT_ULL(7)
#define ICE_DBG_LAN		BIT_ULL(8)
#define ICE_DBG_FLOW		BIT_ULL(9)
#define ICE_DBG_DCB		BIT_ULL(10)
#define ICE_DBG_DIAG		BIT_ULL(11)
#define ICE_DBG_FD		BIT_ULL(12)
#define ICE_DBG_SW		BIT_ULL(13)
#define ICE_DBG_SCHED		BIT_ULL(14)

#ifdef RDMA_SUPPORT
#define ICE_DBG_RDMA		BIT_ULL(15)
#endif
#define ICE_DBG_PKG		BIT_ULL(16)
#define ICE_DBG_RES		BIT_ULL(17)
#ifndef NO_ACL_SUPPORT
#define ICE_DBG_ACL		BIT_ULL(18)
#endif /* !NO_ACL_SUPPORT */
#define ICE_DBG_AQ_MSG		BIT_ULL(24)
#define ICE_DBG_AQ_DESC		BIT_ULL(25)
#define ICE_DBG_AQ_DESC_BUF	BIT_ULL(26)
#define ICE_DBG_AQ_CMD		BIT_ULL(27)
#define ICE_DBG_AQ		(ICE_DBG_AQ_MSG		| \
				 ICE_DBG_AQ_DESC	| \
				 ICE_DBG_AQ_DESC_BUF	| \
				 ICE_DBG_AQ_CMD)

#define ICE_DBG_USER		BIT_ULL(31)
#define ICE_DBG_ALL		0xFFFFFFFFFFFFFFFFULL

#ifndef __ALWAYS_UNUSED
#define __ALWAYS_UNUSED
#endif

#ifndef IS_UNICAST_ETHER_ADDR
#define IS_UNICAST_ETHER_ADDR(addr) \
	((bool)((((u8 *)(addr))[0] % ((u8)0x2)) == 0))
#endif

#ifndef IS_MULTICAST_ETHER_ADDR
#define IS_MULTICAST_ETHER_ADDR(addr) \
	((bool)((((u8 *)(addr))[0] % ((u8)0x2)) == 1))
#endif

#ifndef IS_BROADCAST_ETHER_ADDR
/* Check whether an address is broadcast. */
#define IS_BROADCAST_ETHER_ADDR(addr)	\
	((bool)((((u16 *)(addr))[0] == ((u16)0xffff))))
#endif

#ifndef IS_ZERO_ETHER_ADDR
#define IS_ZERO_ETHER_ADDR(addr) \
	(((bool)((((u16 *)(addr))[0] == ((u16)0x0)))) && \
	 ((bool)((((u16 *)(addr))[1] == ((u16)0x0)))) && \
	 ((bool)((((u16 *)(addr))[2] == ((u16)0x0)))))
#endif

#ifndef IS_ETHER_ADDR_EQUAL
#define IS_ETHER_ADDR_EQUAL(addr1, addr2) \
	(((bool)((((u16 *)(addr1))[0] == ((u16 *)(addr2))[0]))) && \
	 ((bool)((((u16 *)(addr1))[1] == ((u16 *)(addr2))[1]))) && \
	 ((bool)((((u16 *)(addr1))[2] == ((u16 *)(addr2))[2]))))
#endif

enum ice_aq_res_ids {
	ICE_NVM_RES_ID = 1,
	ICE_SPD_RES_ID,
	ICE_CHANGE_LOCK_RES_ID,
	ICE_GLOBAL_CFG_LOCK_RES_ID
};

/* FW update timeout definitions are in milliseconds */
#ifdef FPGA_SUPPORT
#define ICE_NVM_TIMEOUT			360000
#define ICE_CHANGE_LOCK_TIMEOUT		30000
#define ICE_GLOBAL_CFG_LOCK_TIMEOUT	60000
#else
#define ICE_NVM_TIMEOUT			180000
#define ICE_CHANGE_LOCK_TIMEOUT		1000
#ifdef E830_SUPPORT
#define ICE_GLOBAL_CFG_LOCK_TIMEOUT	60000
#else
#define ICE_GLOBAL_CFG_LOCK_TIMEOUT	3000
#endif /* E830_SUPPORT */
#endif

enum ice_aq_res_access_type {
	ICE_RES_READ = 1,
	ICE_RES_WRITE
};

struct ice_driver_ver {
	u8 major_ver;
	u8 minor_ver;
	u8 build_ver;
	u8 subbuild_ver;
	u8 driver_string[32];
};

enum ice_fc_mode {
	ICE_FC_NONE = 0,
	ICE_FC_RX_PAUSE,
	ICE_FC_TX_PAUSE,
	ICE_FC_FULL,
#ifndef LINUX_SUPPORT
	ICE_FC_AUTO,
#endif
	ICE_FC_PFC,
	ICE_FC_DFLT
};

enum ice_phy_cache_mode {
	ICE_FC_MODE = 0,
	ICE_SPEED_MODE,
	ICE_FEC_MODE
};

enum ice_fec_mode {
	ICE_FEC_NONE = 0,
	ICE_FEC_RS,
	ICE_FEC_BASER,
	ICE_FEC_AUTO
};

struct ice_phy_cache_mode_data {
	union {
		enum ice_fec_mode curr_user_fec_req;
		enum ice_fc_mode curr_user_fc_req;
		u16 curr_user_speed_req;
	} data;
};

enum ice_set_fc_aq_failures {
	ICE_SET_FC_AQ_FAIL_NONE = 0,
	ICE_SET_FC_AQ_FAIL_GET,
	ICE_SET_FC_AQ_FAIL_SET,
	ICE_SET_FC_AQ_FAIL_UPDATE
};

/* These are structs for managing the hardware information and the operations */
/* MAC types */
enum ice_mac_type {
	ICE_MAC_UNKNOWN = 0,
#ifdef FPGA_SUPPORT
	ICE_MAC_FPGA,
#endif /* FPGA_SUPPORT */
#ifdef SRIOV_SUPPORT
	ICE_MAC_VF,
#endif /* SRIOV_SUPPORT */
#if defined(E810C_SUPPORT) || defined(E810_XXV_SUPPORT)
	ICE_MAC_E810,
#endif /* E810C_SUPPORT || E810_XXV_SUPPORT */
#ifdef E830_SUPPORT
	ICE_MAC_E830,
#endif /* E830_SUPPORT */
	ICE_MAC_GENERIC,
};


/* Media Types */
enum ice_media_type {
	ICE_MEDIA_UNKNOWN = 0,
	ICE_MEDIA_FIBER,
	ICE_MEDIA_BASET,
	ICE_MEDIA_BACKPLANE,
	ICE_MEDIA_DA,
	ICE_MEDIA_AUI,
};

/* Software VSI types. */
enum ice_vsi_type {
	ICE_VSI_PF = 0,
#ifdef SRIOV_SUPPORT
	ICE_VSI_VF = 1,
#endif /* SRIOV_SUPPORT */
#ifdef VMDQ_SUPPORT
	ICE_VSI_VMDQ2 = 2,
#endif /* VMDQ_SUPPORT */
#ifdef FDIR_SUPPORT
	ICE_VSI_CTRL = 3,	/* equates to ICE_VSI_PF with 1 queue pair */
#endif /* FDIR_SUPPORT */
#ifdef ADQ_SUPPORT
	ICE_VSI_CHNL = 4,
#endif /* ADQ_SUPPORT */
#ifdef OFFLOAD_MACVLAN_SUPPORT
	ICE_VSI_OFFLOAD_MACVLAN = 5,
#endif /* OFFLOAD_MACVLAN_SUPPORT */
	ICE_VSI_LB = 6,
#ifdef ESWITCH_SUPPORT
	ICE_VSI_SWITCHDEV_CTRL = 7,
#endif /* ESWITCH_SUPPORT */
};

struct ice_link_status {
	/* Refer to ice_aq_phy_type for bits definition */
	u64 phy_type_low;
	u64 phy_type_high;
	u8 topo_media_conflict;
	u16 max_frame_size;
	u16 link_speed;
	u16 req_speeds;
	u8 link_cfg_err;
	u8 lse_ena;	/* Link Status Event notification */
	u8 link_info;
	u8 an_info;
	u8 ext_info;
	u8 fec_info;
	u8 pacing;
	/* Refer to #define from module_type[ICE_MODULE_TYPE_TOTAL_BYTE] of
	 * ice_aqc_get_phy_caps structure
	 */
	u8 module_type[ICE_MODULE_TYPE_TOTAL_BYTE];
};

/* Different data queue types: These are mainly for SW consumption. */
enum ice_q {
	ICE_DATA_Q_DOORBELL,
	ICE_DATA_Q_CMPL,
	ICE_DATA_Q_QUANTA,
	ICE_DATA_Q_RX,
	ICE_DATA_Q_TX,
};

/* Different reset sources for which a disable queue AQ call has to be made in
 * order to clean the Tx scheduler as a part of the reset
 */
enum ice_disq_rst_src {
	ICE_NO_RESET = 0,
	ICE_VM_RESET,
#ifdef SRIOV_SUPPORT
	ICE_VF_RESET,
#endif
};

/* PHY info such as phy_type, etc... */
struct ice_phy_info {
	struct ice_link_status link_info;
	struct ice_link_status link_info_old;
	u64 phy_type_low;
	u64 phy_type_high;
	enum ice_media_type media_type;
	u8 get_link_info;
	/* Please refer to struct ice_aqc_get_link_status_data to get
	 * detail of enable bit in curr_user_speed_req
	 */
	u16 curr_user_speed_req;
	enum ice_fec_mode curr_user_fec_req;
	enum ice_fc_mode curr_user_fc_req;
	struct ice_aqc_set_phy_cfg_data curr_user_phy_cfg;
};

#define ICE_MAX_NUM_MIRROR_RULES	64

#if defined(FDIR_SUPPORT) || !defined(NO_ACL_SUPPORT)
/* protocol enumeration for filters */
enum ice_fltr_ptype {
	/* NONE - used for undef/error */
	ICE_FLTR_PTYPE_NONF_NONE = 0,
	ICE_FLTR_PTYPE_NONF_IPV4_UDP,
	ICE_FLTR_PTYPE_NONF_IPV4_TCP,
	ICE_FLTR_PTYPE_NONF_IPV4_SCTP,
	ICE_FLTR_PTYPE_NONF_IPV4_OTHER,
#if defined(DPDK_SUPPORT) || defined(ADV_AVF_SUPPORT)
	ICE_FLTR_PTYPE_NONF_IPV4_GTPU,
	ICE_FLTR_PTYPE_NONF_IPV4_GTPU_EH,
	ICE_FLTR_PTYPE_NONF_IPV4_GTPU_EH_DW,
	ICE_FLTR_PTYPE_NONF_IPV4_GTPU_EH_UP,
	ICE_FLTR_PTYPE_NONF_IPV6_GTPU,
	ICE_FLTR_PTYPE_NONF_IPV6_GTPU_EH,
	ICE_FLTR_PTYPE_NONF_IPV6_GTPU_EH_DW,
	ICE_FLTR_PTYPE_NONF_IPV6_GTPU_EH_UP,
	ICE_FLTR_PTYPE_NONF_IPV4_GTPU_IPV4,
	ICE_FLTR_PTYPE_NONF_IPV4_GTPU_IPV4_UDP,
	ICE_FLTR_PTYPE_NONF_IPV4_GTPU_IPV4_TCP,
	ICE_FLTR_PTYPE_NONF_IPV4_GTPU_IPV6,
	ICE_FLTR_PTYPE_NONF_IPV4_GTPU_IPV6_UDP,
	ICE_FLTR_PTYPE_NONF_IPV4_GTPU_IPV6_TCP,
	ICE_FLTR_PTYPE_NONF_IPV4_GTPU_EH_IPV4,
	ICE_FLTR_PTYPE_NONF_IPV4_GTPU_EH_IPV4_UDP,
	ICE_FLTR_PTYPE_NONF_IPV4_GTPU_EH_IPV4_TCP,
	ICE_FLTR_PTYPE_NONF_IPV4_GTPU_EH_DW_IPV4,
	ICE_FLTR_PTYPE_NONF_IPV4_GTPU_EH_DW_IPV4_UDP,
	ICE_FLTR_PTYPE_NONF_IPV4_GTPU_EH_DW_IPV4_TCP,
	ICE_FLTR_PTYPE_NONF_IPV4_GTPU_EH_UP_IPV4,
	ICE_FLTR_PTYPE_NONF_IPV4_GTPU_EH_UP_IPV4_UDP,
	ICE_FLTR_PTYPE_NONF_IPV4_GTPU_EH_UP_IPV4_TCP,
	ICE_FLTR_PTYPE_NONF_IPV4_GTPU_IPV4_ICMP,
	ICE_FLTR_PTYPE_NONF_IPV4_GTPU_IPV4_OTHER,
	ICE_FLTR_PTYPE_NONF_IPV6_GTPU_IPV6_OTHER,
	ICE_FLTR_PTYPE_NONF_IPV4_GTPU_EH_IPV4_OTHER,
	ICE_FLTR_PTYPE_NONF_IPV6_GTPU_EH_IPV6_OTHER,
	ICE_FLTR_PTYPE_NONF_IPV4_L2TPV3,
	ICE_FLTR_PTYPE_NONF_IPV6_L2TPV3,
	ICE_FLTR_PTYPE_NONF_IPV4_ESP,
	ICE_FLTR_PTYPE_NONF_IPV6_ESP,
	ICE_FLTR_PTYPE_NONF_IPV4_AH,
	ICE_FLTR_PTYPE_NONF_IPV6_AH,
	ICE_FLTR_PTYPE_NONF_IPV4_NAT_T_ESP,
	ICE_FLTR_PTYPE_NONF_IPV6_NAT_T_ESP,
	ICE_FLTR_PTYPE_NONF_IPV4_PFCP_NODE,
	ICE_FLTR_PTYPE_NONF_IPV4_PFCP_SESSION,
	ICE_FLTR_PTYPE_NONF_IPV6_PFCP_NODE,
	ICE_FLTR_PTYPE_NONF_IPV6_PFCP_SESSION,
	ICE_FLTR_PTYPE_NON_IP_L2,
	ICE_FLTR_PTYPE_NONF_ECPRI_TP0,
	ICE_FLTR_PTYPE_NONF_IPV4_UDP_ECPRI_TP0,
	ICE_FLTR_PTYPE_FRAG_IPV4,
	ICE_FLTR_PTYPE_FRAG_IPV6,
#endif /* DPDK_SUPPORT || ADV_AVF_SUPPORT */
	ICE_FLTR_PTYPE_NONF_IPV6_UDP,
	ICE_FLTR_PTYPE_NONF_IPV6_TCP,
	ICE_FLTR_PTYPE_NONF_IPV6_SCTP,
	ICE_FLTR_PTYPE_NONF_IPV6_OTHER,
	ICE_FLTR_PTYPE_NONF_IPV4_UDP_VXLAN,
	ICE_FLTR_PTYPE_MAX,
};

enum ice_fd_hw_seg {
	ICE_FD_HW_SEG_NON_TUN = 0,
	ICE_FD_HW_SEG_TUN,
	ICE_FD_HW_SEG_MAX,
};

#ifndef ADQ_SUPPORT
#ifndef BMSM_MODE
/* 2 VSI = 1 ICE_VSI_PF + 1 ICE_VSI_CTRL */
#define ICE_MAX_FDIR_VSI_PER_FILTER	2
#else /* BMSM_MODE */
/* 9 VSI = 8 ICE_VSI_PF + 1 ICE_VSI_CTRL */
#define ICE_MAX_FDIR_VSI_PER_FILTER	9
#endif /* !BMSM_MODE */
#else
/* 1 ICE_VSI_PF + 1 ICE_VSI_CTRL + ICE_CHNL_MAX_TC */
#define ICE_MAX_FDIR_VSI_PER_FILTER	(2 + ICE_CHNL_MAX_TC)
#endif /* !ADQ_SUPPORT */

struct ice_fd_hw_prof {
	struct ice_flow_seg_info *fdir_seg[ICE_FD_HW_SEG_MAX];
	int cnt;
	u64 entry_h[ICE_MAX_FDIR_VSI_PER_FILTER][ICE_FD_HW_SEG_MAX];
	u16 vsi_h[ICE_MAX_FDIR_VSI_PER_FILTER];
};
#endif /* FDIR_SUPPORT || !NO_ACL_SUPPORT */

/* Common HW capabilities for SW use */
struct ice_hw_common_caps {
	/* Write CSR protection */
	u64 wr_csr_prot;
	u32 switching_mode;
	/* switching mode supported - EVB switching (including cloud) */
#define ICE_NVM_IMAGE_TYPE_EVB		0x0

	/* Manageablity mode & supported protocols over MCTP */
	u32 mgmt_mode;
#define ICE_MGMT_MODE_PASS_THRU_MODE_M		0xF
#define ICE_MGMT_MODE_CTL_INTERFACE_M		0xF0
#define ICE_MGMT_MODE_REDIR_SB_INTERFACE_M	0xF00

	u32 mgmt_protocols_mctp;
#define ICE_MGMT_MODE_PROTO_RSVD	BIT(0)
#define ICE_MGMT_MODE_PROTO_PLDM	BIT(1)
#define ICE_MGMT_MODE_PROTO_OEM		BIT(2)
#define ICE_MGMT_MODE_PROTO_NC_SI	BIT(3)

	u32 os2bmc;
	u32 valid_functions;
#ifndef NO_DCB_SUPPORT
	/* DCB capabilities */
	u32 active_tc_bitmap;
	u32 maxtc;
#endif /* !NO_DCB_SUPPORT */

	/* RSS related capabilities */
	u32 rss_table_size;		/* 512 for PFs and 64 for VFs */
	u32 rss_table_entry_width;	/* RSS Entry width in bits */

	/* Tx/Rx queues */
	u32 num_rxq;			/* Number/Total Rx queues */
	u32 rxq_first_id;		/* First queue ID for Rx queues */
	u32 num_txq;			/* Number/Total Tx queues */
	u32 txq_first_id;		/* First queue ID for Tx queues */

	/* MSI-X vectors */
	u32 num_msix_vectors;
	u32 msix_vector_first_id;

	/* Max MTU for function or device */
	u32 max_mtu;

	/* WOL related */
	u32 num_wol_proxy_fltr;
	u32 wol_proxy_vsi_seid;

	/* LED/SDP pin count */
	u32 led_pin_num;
	u32 sdp_pin_num;

	/* LED/SDP - Supports up to 12 LED pins and 8 SDP signals */
#define ICE_MAX_SUPPORTED_GPIO_LED	12
#define ICE_MAX_SUPPORTED_GPIO_SDP	8
	u8 led[ICE_MAX_SUPPORTED_GPIO_LED];
	u8 sdp[ICE_MAX_SUPPORTED_GPIO_SDP];
#ifdef SRIOV_SUPPORT

	/* SR-IOV virtualization */
	u8 sr_iov_1_1;			/* SR-IOV enabled */
#endif
#if defined(VMDQ_SUPPORT) || defined(OFFLOAD_MACVLAN_SUPPORT) || defined(ADQ_SUPPORT) || defined(ESWITCH_SUPPORT)

	/* VMDQ */
	u8 vmdq;			/* VMDQ supported */
#endif /* VMDQ_SUPPORT || OFFLOAD_MACVLAN_SUPPORT || ADQ_SUPPORT || ESWITCH_SUPPORT */

	/* EVB capabilities */
	u8 evb_802_1_qbg;		/* Edge Virtual Bridging */
	u8 evb_802_1_qbh;		/* Bridge Port Extension */

#ifndef NO_DCB_SUPPORT
	u8 dcb;
#endif
	u8 iscsi;
#ifndef NO_PTP_SUPPORT
	u8 ieee_1588;
#endif
	u8 mgmt_cem;
#ifdef RDMA_SUPPORT
	u8 iwarp;
#endif

	/* WoL and APM support */
#define ICE_WOL_SUPPORT_M		BIT(0)
#define ICE_ACPI_PROG_MTHD_M		BIT(1)
#define ICE_PROXY_SUPPORT_M		BIT(2)
	u8 apm_wol_support;
	u8 acpi_prog_mthd;
	u8 proxy_support;
#if defined(QV_SUPPORT) || defined(PREBOOT_SUPPORT) || defined(LINUX_SUPPORT) || defined(ICE_TDD)
	bool nvm_update_pending_nvm;
	bool nvm_update_pending_orom;
	bool nvm_update_pending_netlist;
#define ICE_NVM_PENDING_NVM_IMAGE		BIT(0)
#define ICE_NVM_PENDING_OROM			BIT(1)
#define ICE_NVM_PENDING_NETLIST			BIT(2)
#endif /* QV_SUPPORT || PREBOOT_SUPPORT || LINUX_SUPPORT || ICE_TDD */
#if defined(QV_SUPPORT) || defined(PREBOOT_SUPPORT)
	bool fw_lockdown_support; /* false - not supported, true - supported */
	bool fw_lockdown_status; /* false - not locked, true - locked */
#define ICE_FW_LOCKDOWN_SUPPORT			BIT(0)
#define ICE_FW_LOCKDOWN_STATUS			BIT(1)
#endif /* QV_SUPPORT || PREBOOT_SUPPORT */
	bool sec_rev_disabled;
	bool update_disabled;
	bool nvm_unified_update;
#define ICE_NVM_MGMT_SEC_REV_DISABLED		BIT(0)
#define ICE_NVM_MGMT_UPDATE_DISABLED		BIT(1)
#define ICE_NVM_MGMT_UNIFIED_UPD_SUPPORT	BIT(3)
#ifdef LEGACY_PREBOOT_SUPPORT
	bool no_drop_policy_support;
#endif /* LEGACY_PREBOOT_SUPPORT */
#if defined(QV_SUPPORT)
	/* Post update reset restriction */
	bool reset_restrict_support; /* false: not supported, true: supported */
#endif /* QV_SUPPORT */

	/* External topology device images within the NVM */
#define ICE_EXT_TOPO_DEV_IMG_COUNT	4
	u32 ext_topo_dev_img_ver_high[ICE_EXT_TOPO_DEV_IMG_COUNT];
	u32 ext_topo_dev_img_ver_low[ICE_EXT_TOPO_DEV_IMG_COUNT];
	u8 ext_topo_dev_img_part_num[ICE_EXT_TOPO_DEV_IMG_COUNT];
#define ICE_EXT_TOPO_DEV_IMG_PART_NUM_S	8
#define ICE_EXT_TOPO_DEV_IMG_PART_NUM_M	\
		MAKEMASK(0xFF, ICE_EXT_TOPO_DEV_IMG_PART_NUM_S)
	bool ext_topo_dev_img_load_en[ICE_EXT_TOPO_DEV_IMG_COUNT];
#define ICE_EXT_TOPO_DEV_IMG_LOAD_EN	BIT(0)
	bool ext_topo_dev_img_prog_en[ICE_EXT_TOPO_DEV_IMG_COUNT];
#define ICE_EXT_TOPO_DEV_IMG_PROG_EN	BIT(1)
};

#ifndef NO_PTP_SUPPORT
/* IEEE 1588 TIME_SYNC specific info */
/* Function specific definitions */
#define ICE_TS_FUNC_ENA_M		BIT(0)
#define ICE_TS_SRC_TMR_OWND_M		BIT(1)
#define ICE_TS_TMR_ENA_M		BIT(2)
#define ICE_TS_TMR_IDX_OWND_S		4
#define ICE_TS_TMR_IDX_OWND_M		BIT(4)
#define ICE_TS_CLK_FREQ_S		16
#define ICE_TS_CLK_FREQ_M		MAKEMASK(0x7, ICE_TS_CLK_FREQ_S)
#define ICE_TS_CLK_SRC_S		20
#define ICE_TS_CLK_SRC_M		BIT(20)
#define ICE_TS_TMR_IDX_ASSOC_S		24
#define ICE_TS_TMR_IDX_ASSOC_M		BIT(24)

struct ice_ts_func_info {
	/* Function specific info */
	u32 clk_freq;
	u8 clk_src;
	u8 tmr_index_assoc;
	u8 ena;
	u8 tmr_index_owned;
	u8 src_tmr_owned;
	u8 tmr_ena;
};

/* Device specific definitions */
#define ICE_TS_TMR0_OWNR_M		0x7
#define ICE_TS_TMR0_OWND_M		BIT(3)
#define ICE_TS_TMR1_OWNR_S		4
#define ICE_TS_TMR1_OWNR_M		MAKEMASK(0x7, ICE_TS_TMR1_OWNR_S)
#define ICE_TS_TMR1_OWND_M		BIT(7)
#define ICE_TS_DEV_ENA_M		BIT(24)
#define ICE_TS_TMR0_ENA_M		BIT(25)
#define ICE_TS_TMR1_ENA_M		BIT(26)

struct ice_ts_dev_info {
	/* Device specific info */
	u32 ena_ports;
	u32 tmr_own_map;
	u32 tmr0_owner;
	u32 tmr1_owner;
	u8 tmr0_owned;
	u8 tmr1_owned;
	u8 ena;
	u8 tmr0_ena;
	u8 tmr1_ena;
};

/* Time Sync Command Definitions */
#define GLTSYN_CMD_INIT_TIME		BIT(0)
#define GLTSYN_CMD_INIT_INCVAL		BIT(1)
#define GLTSYN_CMD_INIT_TIME_INCVAL	(BIT(0) | BIT(1))
#define GLTSYN_CMD_ADJ_TIME		BIT(2)
#define GLTSYN_CMD_ADJ_INIT_TIME	(BIT(2) | BIT(3))
#define GLTSYN_CMD_READ_TIME		BIT(7)
#endif /* !NO_PTP_SUPPORT */

/* Function specific capabilities */
struct ice_hw_func_caps {
	struct ice_hw_common_caps common_cap;
#ifdef SRIOV_SUPPORT
	u32 num_allocd_vfs;		/* Number of allocated VFs */
	u32 vf_base_id;			/* Logical ID of the first VF */
#endif /* SRIOV_SUPPORT */
	u32 guar_num_vsi;
#ifdef FDIR_SUPPORT
	u32 fd_fltr_guar;		/* Number of filters guaranteed */
	u32 fd_fltr_best_effort;	/* Number of best effort filters */
#endif
#ifndef NO_PTP_SUPPORT
	struct ice_ts_func_info ts_func_info;
#endif /* !NO_PTP_SUPPORT */
#ifdef LEGACY_PREBOOT_SUPPORT
	bool no_drop_policy_ena;
#endif /* LEGACY_PREBOOT_SUPPORT */
};

/* Device wide capabilities */
struct ice_hw_dev_caps {
	struct ice_hw_common_caps common_cap;
#ifdef SRIOV_SUPPORT
	u32 num_vfs_exposed;		/* Total number of VFs exposed */
#endif /* SRIOV_SUPPORT */
	u32 num_vsi_allocd_to_host;	/* Excluding EMP VSI */
#ifdef FDIR_SUPPORT
	u32 num_flow_director_fltr;	/* Number of FD filters available */
#endif /* FDIR_SUPPORT */
#ifndef NO_PTP_SUPPORT
	struct ice_ts_dev_info ts_dev_info;
#endif /* !NO_PTP_SUPPORT */
	u32 num_funcs;
};


/* Information about MAC such as address, etc... */
struct ice_mac_info {
	u8 lan_addr[ETH_ALEN];
	u8 perm_addr[ETH_ALEN];
	u8 port_addr[ETH_ALEN];
	u8 wol_addr[ETH_ALEN];
};

/* PCI bus types */
enum ice_bus_type {
	ice_bus_unknown = 0,
	ice_bus_pci_express,
	ice_bus_embedded, /* Is device Embedded versus card */
	ice_bus_reserved
};

/* PCI bus speeds */
enum ice_pcie_bus_speed {
	ice_pcie_speed_unknown	= 0xff,
	ice_pcie_speed_2_5GT	= 0x14,
	ice_pcie_speed_5_0GT	= 0x15,
	ice_pcie_speed_8_0GT	= 0x16,
	ice_pcie_speed_16_0GT	= 0x17
};

/* PCI bus widths */
enum ice_pcie_link_width {
	ice_pcie_lnk_width_resrv	= 0x00,
	ice_pcie_lnk_x1			= 0x01,
	ice_pcie_lnk_x2			= 0x02,
	ice_pcie_lnk_x4			= 0x04,
	ice_pcie_lnk_x8			= 0x08,
	ice_pcie_lnk_x12		= 0x0C,
	ice_pcie_lnk_x16		= 0x10,
	ice_pcie_lnk_x32		= 0x20,
	ice_pcie_lnk_width_unknown	= 0xff,
};

/* Reset types used to determine which kind of reset was requested. These
 * defines match what the RESET_TYPE field of the GLGEN_RSTAT register.
 * ICE_RESET_PFR does not match any RESET_TYPE field in the GLGEN_RSTAT register
 * because its reset source is different than the other types listed.
 */
enum ice_reset_req {
	ICE_RESET_POR	= 0,
	ICE_RESET_INVAL	= 0,
	ICE_RESET_CORER	= 1,
	ICE_RESET_GLOBR	= 2,
	ICE_RESET_EMPR	= 3,
	ICE_RESET_PFR	= 4,
};

/* Bus parameters */
struct ice_bus_info {
	enum ice_pcie_bus_speed speed;
	enum ice_pcie_link_width width;
	enum ice_bus_type type;
	u16 domain_num;
	u16 device;
	u8 func;
	u8 bus_num;
};

/* Flow control (FC) parameters */
struct ice_fc_info {
	enum ice_fc_mode current_mode;	/* FC mode in effect */
	enum ice_fc_mode req_mode;	/* FC mode requested by caller */
};

/* Option ROM version information */
struct ice_orom_info {
	u8 major;			/* Major version of OROM */
	u8 patch;			/* Patch version of OROM */
	u16 build;			/* Build version of OROM */
	u32 srev;			/* Security revision */
};

/* NVM version information */
struct ice_nvm_info {
	u32 eetrack;
	u32 srev;
	u8 major;
	u8 minor;
};

/* Minimum Security Revision information */
struct ice_minsrev_info {
	u32 nvm;
	u32 orom;
	u8 nvm_valid : 1;
	u8 orom_valid : 1;
};

#if !defined(DPDK_SUPPORT) || defined(AE_DRIVER)
/* netlist version information */
struct ice_netlist_info {
	u32 major;			/* major high/low */
	u32 minor;			/* minor high/low */
	u32 type;			/* type high/low */
	u32 rev;			/* revision high/low */
	u32 hash;			/* SHA-1 hash word */
	u16 cust_ver;			/* customer version */
};
#endif /* !DPDK_SUPPORT || AE_DRIVER */

/* Enumeration of possible flash banks for the NVM, OROM, and Netlist modules
 * of the flash image.
 */
enum ice_flash_bank {
	ICE_INVALID_FLASH_BANK,
	ICE_1ST_FLASH_BANK,
	ICE_2ND_FLASH_BANK,
};

/* Enumeration of which flash bank is desired to read from, either the active
 * bank or the inactive bank. Used to abstract 1st and 2nd bank notion from
 * code which just wants to read the active or inactive flash bank.
 */
enum ice_bank_select {
	ICE_ACTIVE_FLASH_BANK,
	ICE_INACTIVE_FLASH_BANK,
};

/* information for accessing NVM, OROM, and Netlist flash banks */
struct ice_bank_info {
	u32 nvm_ptr;				/* Pointer to 1st NVM bank */
	u32 nvm_size;				/* Size of NVM bank */
	u32 orom_ptr;				/* Pointer to 1st OROM bank */
	u32 orom_size;				/* Size of OROM bank */
	u32 netlist_ptr;			/* Pointer to 1st Netlist bank */
	u32 netlist_size;			/* Size of Netlist bank */
	enum ice_flash_bank nvm_bank;		/* Active NVM bank */
	enum ice_flash_bank orom_bank;		/* Active OROM bank */
	enum ice_flash_bank netlist_bank;	/* Active Netlist bank */
};

/* Flash Chip Information */
struct ice_flash_info {
	struct ice_orom_info orom;	/* Option ROM version info */
	struct ice_nvm_info nvm;	/* NVM version information */
#if !defined(DPDK_SUPPORT) || defined(AE_DRIVER)
	struct ice_netlist_info netlist;/* Netlist version info */
#endif
	struct ice_bank_info banks;	/* Flash Bank information */
	u16 sr_words;			/* Shadow RAM size in words */
	u32 flash_size;			/* Size of available flash in bytes */
	u8 blank_nvm_mode;		/* is NVM empty (no FW present) */
};

struct ice_link_default_override_tlv {
	u8 options;
#define ICE_LINK_OVERRIDE_OPT_M		0x3F
#define ICE_LINK_OVERRIDE_STRICT_MODE	BIT(0)
#define ICE_LINK_OVERRIDE_EPCT_DIS	BIT(1)
#define ICE_LINK_OVERRIDE_PORT_DIS	BIT(2)
#define ICE_LINK_OVERRIDE_EN		BIT(3)
#define ICE_LINK_OVERRIDE_AUTO_LINK_DIS	BIT(4)
#define ICE_LINK_OVERRIDE_EEE_EN	BIT(5)
	u8 phy_config;
#define ICE_LINK_OVERRIDE_PHY_CFG_S	8
#define ICE_LINK_OVERRIDE_PHY_CFG_M	(0xC3 << ICE_LINK_OVERRIDE_PHY_CFG_S)
#define ICE_LINK_OVERRIDE_PAUSE_M	0x3
#define ICE_LINK_OVERRIDE_LESM_EN	BIT(6)
#define ICE_LINK_OVERRIDE_AUTO_FEC_EN	BIT(7)
	u8 fec_options;
#define ICE_LINK_OVERRIDE_FEC_OPT_M	0xFF
	u8 rsvd1;
	u64 phy_type_low;
	u64 phy_type_high;
};

#define ICE_NVM_VER_LEN	32

/* Max number of port to queue branches w.r.t topology */
#define ICE_TXSCHED_MAX_BRANCHES ICE_MAX_TRAFFIC_CLASS

#define ice_for_each_traffic_class(_i)	\
	for ((_i) = 0; (_i) < ICE_MAX_TRAFFIC_CLASS; (_i)++)

/* ICE_DFLT_AGG_ID means that all new VM(s)/VSI node connects
 * to driver defined policy for default aggregator
 */
#define ICE_INVAL_TEID 0xFFFFFFFF
#define ICE_DFLT_AGG_ID 0

struct ice_sched_node {
	struct ice_sched_node *parent;
	struct ice_sched_node *sibling; /* next sibling in the same layer */
	struct ice_sched_node **children;
	struct ice_aqc_txsched_elem_data info;
	u32 agg_id;			/* aggregator group ID */
	u16 vsi_handle;
	u8 in_use;			/* suspended or in use */
	u8 tx_sched_layer;		/* Logical Layer (1-9) */
	u8 num_children;
	u8 tc_num;
	u8 owner;
#ifdef BMSM_MODE
	u8 cgd;				/* Valid only for Layer 2 */
#endif
#define ICE_SCHED_NODE_OWNER_LAN	0
#define ICE_SCHED_NODE_OWNER_AE		1
#define ICE_SCHED_NODE_OWNER_RDMA	2
#ifdef BMSM_MODE
#define ICE_MAX_CGD_PER_PORT		4
#endif
};

/* Access Macros for Tx Sched Elements data */
#define ICE_TXSCHED_GET_NODE_TEID(x) LE32_TO_CPU((x)->info.node_teid)
#define ICE_TXSCHED_GET_PARENT_TEID(x) LE32_TO_CPU((x)->info.parent_teid)
#define ICE_TXSCHED_GET_CIR_RL_ID(x)	\
	LE16_TO_CPU((x)->info.cir_bw.bw_profile_idx)
#define ICE_TXSCHED_GET_EIR_RL_ID(x)	\
	LE16_TO_CPU((x)->info.eir_bw.bw_profile_idx)
#define ICE_TXSCHED_GET_SRL_ID(x) LE16_TO_CPU((x)->info.srl_id)
#define ICE_TXSCHED_GET_CIR_BWALLOC(x)	\
	LE16_TO_CPU((x)->info.cir_bw.bw_alloc)
#define ICE_TXSCHED_GET_EIR_BWALLOC(x)	\
	LE16_TO_CPU((x)->info.eir_bw.bw_alloc)

#ifndef NO_UNUSED_SCHED_CODE
struct ice_sched_rl_profile {
	u32 rate; /* In Kbps */
	struct ice_aqc_rl_profile_elem info;
};
#endif /* !NO_UNUSED_SCHED_CODE */

/* The aggregator type determines if identifier is for a VSI group,
 * aggregator group, aggregator of queues, or queue group.
 */
enum ice_agg_type {
	ICE_AGG_TYPE_UNKNOWN = 0,
	ICE_AGG_TYPE_TC,
	ICE_AGG_TYPE_AGG, /* aggregator */
	ICE_AGG_TYPE_VSI,
	ICE_AGG_TYPE_QG,
	ICE_AGG_TYPE_Q
};

/* Rate limit types */
enum ice_rl_type {
	ICE_UNKNOWN_BW = 0,
	ICE_MIN_BW,		/* for CIR profile */
	ICE_MAX_BW,		/* for EIR profile */
	ICE_SHARED_BW		/* for shared profile */
};

#define ICE_SCHED_MIN_BW		500		/* in Kbps */
#define ICE_SCHED_MAX_BW		100000000	/* in Kbps */
#define ICE_SCHED_DFLT_BW		0xFFFFFFFF	/* unlimited */
#define ICE_SCHED_NO_PRIORITY		0
#define ICE_SCHED_NO_BW_WT		0
#define ICE_SCHED_DFLT_RL_PROF_ID	0
#define ICE_SCHED_NO_SHARED_RL_PROF_ID	0xFFFF
#define ICE_SCHED_DFLT_BW_WT		4
#define ICE_SCHED_INVAL_PROF_ID		0xFFFF
#define ICE_SCHED_DFLT_BURST_SIZE	(15 * 1024)	/* in bytes (15k) */

#ifndef NO_UNUSED_SCHED_CODE
/* Access Macros for Tx Sched RL Profile data */
#define ICE_TXSCHED_GET_RL_PROF_ID(p) LE16_TO_CPU((p)->info.profile_id)
#define ICE_TXSCHED_GET_RL_MBS(p) LE16_TO_CPU((p)->info.max_burst_size)
#define ICE_TXSCHED_GET_RL_MULTIPLIER(p) LE16_TO_CPU((p)->info.rl_multiply)
#define ICE_TXSCHED_GET_RL_WAKEUP_MV(p) LE16_TO_CPU((p)->info.wake_up_calc)
#define ICE_TXSCHED_GET_RL_ENCODE(p) LE16_TO_CPU((p)->info.rl_encode)

#endif
#define ICE_MAX_PORT_PER_PCI_DEV	8

/* The following tree example shows the naming conventions followed under
 * ice_port_info struct for default scheduler tree topology.
 *
 *                 A tree on a port
 *                       *                ---> root node
 *        (TC0)/  /  /  / \  \  \  \(TC7) ---> num_branches (range:1- 8)
 *            *  *  *  *   *  *  *  *     |
 *           /                            |
 *          *                             |
 *         /                              |-> num_elements (range:1 - 9)
 *        *                               |   implies num_of_layers
 *       /                                |
 *   (a)*                                 |
 *
 *  (a) is the last_node_teid(not of type Leaf). A leaf node is created under
 *  (a) as child node where queues get added, add Tx/Rx queue admin commands;
 *  need TEID of (a) to add queues.
 *
 *  This tree
 *       -> has 8 branches (one for each TC)
 *       -> First branch (TC0) has 4 elements
 *       -> has 4 layers
 *       -> (a) is the topmost layer node created by firmware on branch 0
 *
 *  Note: Above asterisk tree covers only basic terminology and scenario.
 *  Refer to the documentation for more info.
 */

 /* Data structure for saving BW information */
enum ice_bw_type {
	ICE_BW_TYPE_PRIO,
	ICE_BW_TYPE_CIR,
	ICE_BW_TYPE_CIR_WT,
	ICE_BW_TYPE_EIR,
	ICE_BW_TYPE_EIR_WT,
	ICE_BW_TYPE_SHARED,
	ICE_BW_TYPE_CNT		/* This must be last */
};

struct ice_bw {
	u32 bw;
	u16 bw_alloc;
};

struct ice_bw_type_info {
	ice_declare_bitmap(bw_t_bitmap, ICE_BW_TYPE_CNT);
	u8 generic;
	struct ice_bw cir_bw;
	struct ice_bw eir_bw;
	u32 shared_bw;
};

/* VSI queue context structure for given TC */
struct ice_q_ctx {
	u16  q_handle;
	u32  q_teid;
	/* bw_t_info saves queue BW information */
	struct ice_bw_type_info bw_t_info;
};

/* VSI type list entry to locate corresponding VSI/aggregator nodes */
struct ice_sched_vsi_info {
	struct ice_sched_node *vsi_node[ICE_MAX_TRAFFIC_CLASS];
	struct ice_sched_node *ag_node[ICE_MAX_TRAFFIC_CLASS];
	u16 max_lanq[ICE_MAX_TRAFFIC_CLASS];
#ifdef RDMA_SUPPORT
	u16 max_rdmaq[ICE_MAX_TRAFFIC_CLASS];
#endif /* RDMA_SUPPORT */
	/* bw_t_info saves VSI BW information */
	struct ice_bw_type_info bw_t_info[ICE_MAX_TRAFFIC_CLASS];
};

#if !defined(NO_DCB_SUPPORT) || defined(ADQ_SUPPORT)
/* CEE or IEEE 802.1Qaz ETS Configuration data */
struct ice_dcb_ets_cfg {
	u8 willing;
	u8 cbs;
	u8 maxtcs;
	u8 prio_table[ICE_MAX_TRAFFIC_CLASS];
	u8 tcbwtable[ICE_MAX_TRAFFIC_CLASS];
	u8 tsatable[ICE_MAX_TRAFFIC_CLASS];
};

/* CEE or IEEE 802.1Qaz PFC Configuration data */
struct ice_dcb_pfc_cfg {
	u8 willing;
	u8 mbc;
	u8 pfccap;
	u8 pfcena;
};

/* CEE or IEEE 802.1Qaz Application Priority data */
struct ice_dcb_app_priority_table {
	u16 prot_id;
	u8 priority;
	u8 selector;
};

#define ICE_MAX_USER_PRIORITY		8
#define ICE_DCBX_MAX_APPS		64
#define ICE_DSCP_NUM_VAL		64
#define ICE_LLDPDU_SIZE			1500
#define ICE_TLV_STATUS_OPER		0x1
#define ICE_TLV_STATUS_SYNC		0x2
#define ICE_TLV_STATUS_ERR		0x4
#ifndef ICE_APP_PROT_ID_FCOE
#define ICE_APP_PROT_ID_FCOE		0x8906
#endif /* ICE_APP_PROT_ID_FCOE */
#ifndef ICE_APP_PROT_ID_ISCSI
#define ICE_APP_PROT_ID_ISCSI		0x0cbc
#endif /* ICE_APP_PROT_ID_ISCSI */
#ifndef ICE_APP_PROT_ID_ISCSI_860
#define ICE_APP_PROT_ID_ISCSI_860	0x035c
#endif /* ICE_APP_PROT_ID_ISCSI_860 */
#ifndef ICE_APP_PROT_ID_FIP
#define ICE_APP_PROT_ID_FIP		0x8914
#endif /* ICE_APP_PROT_ID_FIP */
#define ICE_APP_SEL_ETHTYPE		0x1
#define ICE_APP_SEL_TCPIP		0x2
#define ICE_CEE_APP_SEL_ETHTYPE		0x0
#define ICE_CEE_APP_SEL_TCPIP		0x1

struct ice_dcbx_cfg {
	u32 numapps;
	u32 tlv_status; /* CEE mode TLV status */
	struct ice_dcb_ets_cfg etscfg;
	struct ice_dcb_ets_cfg etsrec;
	struct ice_dcb_pfc_cfg pfc;
#define ICE_QOS_MODE_VLAN	0x0
#define ICE_QOS_MODE_DSCP	0x1
	u8 pfc_mode;
	struct ice_dcb_app_priority_table app[ICE_DCBX_MAX_APPS];
	/* when DSCP mapping defined by user set its bit to 1 */
	ice_declare_bitmap(dscp_mapped, ICE_DSCP_NUM_VAL);
	/* array holding DSCP -> UP/TC values for DSCP L3 QoS mode */
	u8 dscp_map[ICE_DSCP_NUM_VAL];
	u8 dcbx_mode;
#define ICE_DCBX_MODE_CEE	0x1
#define ICE_DCBX_MODE_IEEE	0x2
	u8 app_mode;
#define ICE_DCBX_APPS_NON_WILLING	0x1
};

struct ice_qos_cfg {
	struct ice_dcbx_cfg local_dcbx_cfg;	/* Oper/Local Cfg */
#ifndef NO_DCB_SUPPORT
	struct ice_dcbx_cfg desired_dcbx_cfg;	/* CEE Desired Cfg */
	struct ice_dcbx_cfg remote_dcbx_cfg;	/* Peer Cfg */
	u8 dcbx_status : 3;			/* see ICE_DCBX_STATUS_DIS */
	u8 is_sw_lldp : 1;
#endif /* !NO_DCB_SUPPORT */
};
#endif /* !NO_DCB_SUPPORT || ADQ_SUPPORT */

struct ice_port_info {
	struct ice_sched_node *root;	/* Root Node per Port */
	struct ice_hw *hw;		/* back pointer to HW instance */
	u32 last_node_teid;		/* scheduler last node info */
	u16 sw_id;			/* Initial switch ID belongs to port */
	u16 pf_vf_num;
	u8 port_state;
#define ICE_SCHED_PORT_STATE_INIT	0x0
#define ICE_SCHED_PORT_STATE_READY	0x1
	u8 lport;
#define ICE_LPORT_MASK			0xff
	u16 dflt_tx_vsi_rule_id;
	u16 dflt_tx_vsi_num;
	u16 dflt_rx_vsi_rule_id;
	u16 dflt_rx_vsi_num;
#if defined(QV_SUPPORT)
	u16 dflt_vsi_rule_id;
	u16 dflt_vsi_num;
#endif /* (SWITCH_MODE && !BMSM_MODE) || QV_SUPPORT */
	struct ice_fc_info fc;
	struct ice_mac_info mac;
	struct ice_phy_info phy;
	struct ice_lock sched_lock;	/* protect access to TXSched tree */
	struct ice_sched_node *
		sib_head[ICE_MAX_TRAFFIC_CLASS][ICE_AQC_TOPO_MAX_LEVEL_NUM];
	struct ice_bw_type_info root_node_bw_t_info;
	struct ice_bw_type_info tc_node_bw_t_info[ICE_MAX_TRAFFIC_CLASS];
#if !defined(NO_DCB_SUPPORT) || defined(ADQ_SUPPORT)
	struct ice_qos_cfg qos_cfg;
#endif /* !NO_DCB_SUPPORT */
	u8 is_vf:1;
};

struct ice_switch_info {
	struct LIST_HEAD_TYPE vsi_list_map_head;
	struct ice_sw_recipe *recp_list;
	u16 prof_res_bm_init;
	u16 max_used_prof_index;

	ice_declare_bitmap(prof_res_bm[ICE_MAX_NUM_PROFILES], ICE_MAX_FV_WORDS);
};


#ifdef SRIOV_SUPPORT
/* Enum defining the different states of the mailbox snapshot in the
 * PF-VF mailbox overflow detection algorithm. The snapshot can be in
 * states:
 * 1. ICE_MAL_VF_DETECT_STATE_NEW_SNAPSHOT - generate a new static snapshot
 * within the mailbox buffer.
 * 2. ICE_MAL_VF_DETECT_STATE_TRAVERSE - iterate through the mailbox snaphot
 * 3. ICE_MAL_VF_DETECT_STATE_DETECT - track the messages sent per VF via the
 * mailbox and mark any VFs sending more messages than the threshold limit set.
 * 4. ICE_MAL_VF_DETECT_STATE_INVALID - Invalid mailbox state set to 0xFFFFFFFF.
 */
enum ice_mbx_snapshot_state {
	ICE_MAL_VF_DETECT_STATE_NEW_SNAPSHOT = 0,
	ICE_MAL_VF_DETECT_STATE_TRAVERSE,
	ICE_MAL_VF_DETECT_STATE_DETECT,
	ICE_MAL_VF_DETECT_STATE_INVALID = 0xFFFFFFFF,
};

/* Structure to hold information of the static snapshot and the mailbox
 * buffer data used to generate and track the snapshot.
 * 1. state: the state of the mailbox snapshot in the malicious VF
 * detection state handler ice_mbx_vf_state_handler()
 * 2. head : head of the mailbox snapshot in a circular mailbox buffer
 * 3. tail : tail of the mailbox snapshot in a circular mailbox buffer
 * 4. num_iterations: number of messages traversed in circular mailbox buffer
 * 5. num_msg_proc: number of messages processed in mailbox
 * 6. num_pending_arq: number of pending asynchronous messages
 * 7. max_num_msgs_mbx: maximum messages in mailbox for currently
 * serviced work item or interrupt.
 */
struct ice_mbx_snap_buffer_data {
	enum ice_mbx_snapshot_state state;
	u32 head;
	u32 tail;
	u32 num_iterations;
	u16 num_msg_proc;
	u16 num_pending_arq;
	u16 max_num_msgs_mbx;
};

/* Structure to track messages sent by VFs on mailbox:
 * 1. vf_cntr : a counter array of VFs to track the number of
 * asynchronous messages sent by each VF
 * 2. vfcntr_len : number of entries in VF counter array
 */
struct ice_mbx_vf_counter {
	u32 *vf_cntr;
	u32 vfcntr_len;
};

/* Structure to hold data relevant to the captured static snapshot
 * of the PF-VF mailbox.
 */
struct ice_mbx_snapshot {
	struct ice_mbx_snap_buffer_data mbx_buf;
	struct ice_mbx_vf_counter mbx_vf;
};

/* Structure to hold data to be used for capturing or updating a
 * static snapshot.
 * 1. num_msg_proc: number of messages processed in mailbox
 * 2. num_pending_arq: number of pending asynchronous messages
 * 3. max_num_msgs_mbx: maximum messages in mailbox for currently
 * serviced work item or interrupt.
 * 4. async_watermark_val: An upper threshold set by caller to determine
 * if the pending arq count is large enough to assume that there is
 * the possibility of a mailicious VF.
 */
struct ice_mbx_data {
	u16 num_msg_proc;
	u16 num_pending_arq;
	u16 max_num_msgs_mbx;
	u16 async_watermark_val;
};
#endif /* SRIOV_SUPPORT */

/* Port hardware description */
struct ice_hw {
#ifdef LEGACY_PREBOOT_SUPPORT
	/* must be at the top for rd/wr32 to work properly in some contexts */
	u16 bdf; /* PCI B:D:F in format passed by legacy BIOS */
#endif /* LEGACY_PREBOOT_SUPPORT */
	u8 *hw_addr;
	void *back;
	struct ice_aqc_layer_props *layer_info;
	struct ice_port_info *port_info;
#ifndef NO_UNUSED_SCHED_CODE
	/* 2D Array for each Tx Sched RL Profile type */
	struct ice_sched_rl_profile **cir_profiles;
	struct ice_sched_rl_profile **eir_profiles;
	struct ice_sched_rl_profile **srl_profiles;
#endif
	/* PSM clock frequency for calculating RL profile params */
	u32 psm_clk_freq;
	u64 debug_mask;		/* BITMAP for debug mask */
	enum ice_mac_type mac_type;

#if FDIR_SUPPORT
	u16 fd_ctr_base;	/* FD counter base index */
#endif /* FDIR_SUPPORT */
#if defined(PREBOOT_SUPPORT) || defined(QV_SUPPORT)
	u16 fw_vsi_num;
#endif /* PREBOOT_SUPPORT || QV_SUPPORT */
	/* pci info */
	u16 device_id;
	u16 vendor_id;
	u16 subsystem_device_id;
	u16 subsystem_vendor_id;
	u8 revision_id;

	u8 pf_id;		/* device profile info */

	u16 max_burst_size;	/* driver sets this value */

	/* Tx Scheduler values */
	u8 num_tx_sched_layers;
	u8 num_tx_sched_phys_layers;
	u8 flattened_layers;
	u8 max_cgds;
	u8 sw_entry_point_layer;
	u16 max_children[ICE_AQC_TOPO_MAX_LEVEL_NUM];
	struct LIST_HEAD_TYPE agg_list;	/* lists all aggregator */
	/* List contain profile ID(s) and other params per layer */
	struct LIST_HEAD_TYPE rl_prof_list[ICE_AQC_TOPO_MAX_LEVEL_NUM];
	struct ice_vsi_ctx *vsi_ctx[ICE_MAX_VSI];
#ifndef BMSM_MODE
	u8 evb_veb;		/* true for VEB, false for VEPA */
#else
	u8 evb_veb[ICE_MAX_PORT_PER_PCI_DEV];
#endif /* !BMSM_MODE */
	u8 reset_ongoing;	/* true if HW is in reset, false otherwise */
	struct ice_bus_info bus;
	struct ice_flash_info flash;
	struct ice_hw_dev_caps dev_caps;	/* device capabilities */
	struct ice_hw_func_caps func_caps;	/* function capabilities */

	struct ice_switch_info *switch_info;	/* switch filter lists */

	/* Control Queue info */
#if defined(PF_DRIVER) || defined(INTEGRATED_VF) || defined(AE_DRIVER)
	struct ice_ctl_q_info adminq;
#endif
#ifndef NO_SBQ_SUPPORT
#if defined(PF_DRIVER) || defined(INTEGRATED_VF)
	struct ice_ctl_q_info sbq;
#endif
#endif /* !NO_SBQ_SUPPORT */
#ifndef NO_MBXQ_SUPPORT
	struct ice_ctl_q_info mailboxq;
#ifdef QV_SUPPORT
	struct ice_ctl_q_info dsc_fw_q;
	struct ice_ctl_q_info hlp_fw_q;
	struct ice_ctl_q_info hlp_sb_q;
	struct ice_ctl_q_info ipseq_sb_q;
#endif /* QV_SUPPORT */
#endif /* NO_MBXQ_SUPPORT */
#ifdef QV_SUPPORT
	bool toolsq_in_csr;
	bool traffic_init;
	bool cnv_support;
#endif /* QV_SUPPORT */
#ifdef DPDK_SUPPORT
	/* Additional function to send AdminQ command */
	int (*aq_send_cmd_fn)(void *param, struct ice_aq_desc *desc,
			      void *buf, u16 buf_size);
	void *aq_send_cmd_param;
	u8 dcf_enabled;		/* Device Config Function */
#endif /* DPDK_SUPPORT */
#if defined(LINUX_SUPPORT) && defined(DCF_SUPPORT)
#define DCF_ACL_CAP		0x01	/* DCF ACL capability */
#define DCF_UDP_TUNNEL_CAP	0x02	/* DCF UDP Tunnel capability */
	u8 dcf_caps;
#endif
	u8 api_branch;		/* API branch version */
	u8 api_maj_ver;		/* API major version */
	u8 api_min_ver;		/* API minor version */
	u8 api_patch;		/* API patch version */
	u8 fw_branch;		/* firmware branch version */
	u8 fw_maj_ver;		/* firmware major version */
	u8 fw_min_ver;		/* firmware minor version */
	u8 fw_patch;		/* firmware patch version */
	u32 fw_build;		/* firmware build number */

#ifdef FWLOG_SUPPORT_V2
	struct ice_fwlog_cfg fwlog_cfg;
	bool fwlog_support_ena; /* does hardware support FW logging? */
#endif /* FWLOG_SUPPORT_V2 */

/* Device max aggregate bandwidths corresponding to the GL_PWR_MODE_CTL
 * register. Used for determining the ITR/INTRL granularity during
 * initialization.
 */
#define ICE_MAX_AGG_BW_200G	0x0
#define ICE_MAX_AGG_BW_100G	0X1
#define ICE_MAX_AGG_BW_50G	0x2
#define ICE_MAX_AGG_BW_25G	0x3
	/* ITR granularity for different speeds */
#define ICE_ITR_GRAN_ABOVE_25	2
#define ICE_ITR_GRAN_MAX_25	4
	/* ITR granularity in 1 us */
	u8 itr_gran;
	/* INTRL granularity for different speeds */
#define ICE_INTRL_GRAN_ABOVE_25	4
#define ICE_INTRL_GRAN_MAX_25	8
	/* INTRL granularity in 1 us */
	u8 intrl_gran;

	u8 ucast_shared;	/* true if VSIs can share unicast addr */

#define ICE_PHY_PER_NAC		1
#define ICE_MAX_QUAD		2
#define ICE_NUM_QUAD_TYPE	2
#define ICE_PORTS_PER_QUAD	4
#define ICE_PHY_0_LAST_QUAD	1
#define ICE_PORTS_PER_PHY	8
#define ICE_NUM_EXTERNAL_PORTS		ICE_PORTS_PER_PHY


#ifndef NO_FLEXP_SUPPORT
	/* Active package version (currently active) */
	struct ice_pkg_ver active_pkg_ver;
	u32 active_track_id;
	u8 active_pkg_name[ICE_PKG_NAME_SIZE];
	u8 active_pkg_in_nvm;

	enum ice_aq_err pkg_dwnld_status;

	/* Driver's package ver - (from the Ice Metadata section) */
	struct ice_pkg_ver pkg_ver;
	u8 pkg_name[ICE_PKG_NAME_SIZE];

	/* Driver's Ice segment format version and id (from the Ice seg) */
	struct ice_pkg_ver ice_seg_fmt_ver;
	u8 ice_seg_id[ICE_SEG_ID_SIZE];

	/* Pointer to the ice segment */
	struct ice_seg *seg;

	/* Pointer to allocated copy of pkg memory */
	u8 *pkg_copy;
	u32 pkg_size;

	/* tunneling info */
	struct ice_lock tnl_lock;
	struct ice_tunnel_table tnl;
#ifdef DVM_SUPPORT

	/* dvm boost update information */
	struct ice_dvm_table dvm_upd;
#endif /* DVM_SUPPORT */
#endif /* !NO_FLEXP_SUPPORT */

#if (defined(SIMICS_SUPPORT) || defined(FPGA_SUPPORT)) && defined(SV_SUPPORT)
	struct ice_lock test_lock;	/* for osdep lock test */
#endif
#ifdef FPGA_SUPPORT
	u8 is_fpga;
#endif
#ifdef QV_SUPPORT
	/* Debug enable for QV use */
	u8 dbg_ena;
	u8 qv_force_init;
	u8 is_switch_mode;
	u8 sv_dbg_ena;
#endif /* QV_SUPPORT */
#ifndef NO_ACL_SUPPORT
	struct ice_acl_tbl *acl_tbl;
	struct ice_fd_hw_prof **acl_prof;
	u16 acl_fltr_cnt[ICE_FLTR_PTYPE_MAX];
#endif /* !NO_ACL_SUPPORT */
#ifndef NO_PF_RESET_DISABLE
	u8 pf_reset_dis; /* set to avoid PF reset during init */
#endif
#ifndef NO_FLEXP_SUPPORT
	/* HW block tables */
	struct ice_blk_info blk[ICE_BLK_COUNT];
	struct ice_lock fl_profs_locks[ICE_BLK_COUNT];	/* lock fltr profiles */
	struct LIST_HEAD_TYPE fl_profs[ICE_BLK_COUNT];
#endif
#ifdef FDIR_SUPPORT
	/* Flow Director filter info */
	int fdir_active_fltr;

	struct ice_lock fdir_fltr_lock;	/* protect Flow Director */
	struct LIST_HEAD_TYPE fdir_list_head;

	/* Book-keeping of side-band filter count per flow-type.
	 * This is used to detect and handle input set changes for
	 * respective flow-type.
	 */
	u16 fdir_fltr_cnt[ICE_FLTR_PTYPE_MAX];

	struct ice_fd_hw_prof **fdir_prof;
	ice_declare_bitmap(fdir_perfect_fltr, ICE_FLTR_PTYPE_MAX);
#endif /* FDIR_SUPPORT */
#ifndef NO_FLEXP_SUPPORT
	struct ice_lock rss_locks;	/* protect RSS configuration */
	struct LIST_HEAD_TYPE rss_list_head;
#endif /* NO_FLEXP_SUPPORT */
#ifdef SRIOV_SUPPORT
	struct ice_mbx_snapshot mbx_snapshot;
#endif /* SRIOV_SUPPORT */
#if defined(DPDK_SUPPORT) || defined(ADV_AVF_SUPPORT)
	ice_declare_bitmap(hw_ptype, ICE_FLOW_PTYPE_MAX);
#endif /* DPDK_SUPPORT || ADV_AVF_SUPPORT */
	u8 dvm_ena;
};

/* Statistics collected by each port, VSI, VEB, and S-channel */
struct ice_eth_stats {
	u64 rx_bytes;			/* gorc */
	u64 rx_unicast;			/* uprc */
	u64 rx_multicast;		/* mprc */
	u64 rx_broadcast;		/* bprc */
	u64 rx_discards;		/* rdpc */
	u64 rx_unknown_protocol;	/* rupp */
	u64 tx_bytes;			/* gotc */
	u64 tx_unicast;			/* uptc */
	u64 tx_multicast;		/* mptc */
	u64 tx_broadcast;		/* bptc */
	u64 tx_discards;		/* tdpc */
	u64 tx_errors;			/* tepc */
#ifndef LINUX_SUPPORT
	u64 rx_no_desc;			/* repc */
	u64 rx_errors;			/* repc */
#endif
};

#define ICE_MAX_UP	8

/* Statistics collected per VEB per User Priority (UP) for up to 8 UPs */
struct ice_veb_up_stats {
	u64 up_rx_pkts[ICE_MAX_UP];
	u64 up_rx_bytes[ICE_MAX_UP];
	u64 up_tx_pkts[ICE_MAX_UP];
	u64 up_tx_bytes[ICE_MAX_UP];
};

/* Statistics collected by the MAC */
struct ice_hw_port_stats {
	/* eth stats collected by the port */
	struct ice_eth_stats eth;
	/* additional port specific stats */
	u64 tx_dropped_link_down;	/* tdold */
	u64 crc_errors;			/* crcerrs */
	u64 illegal_bytes;		/* illerrc */
	u64 error_bytes;		/* errbc */
	u64 mac_local_faults;		/* mlfc */
	u64 mac_remote_faults;		/* mrfc */
	u64 rx_len_errors;		/* rlec */
	u64 link_xon_rx;		/* lxonrxc */
	u64 link_xoff_rx;		/* lxoffrxc */
	u64 link_xon_tx;		/* lxontxc */
	u64 link_xoff_tx;		/* lxofftxc */
#ifndef NO_DCB_SUPPORT
	u64 priority_xon_rx[8];		/* pxonrxc[8] */
	u64 priority_xoff_rx[8];	/* pxoffrxc[8] */
	u64 priority_xon_tx[8];		/* pxontxc[8] */
	u64 priority_xoff_tx[8];	/* pxofftxc[8] */
	u64 priority_xon_2_xoff[8];	/* pxon2offc[8] */
#endif /* !NO_DCB_SUPPORT */
	u64 rx_size_64;			/* prc64 */
	u64 rx_size_127;		/* prc127 */
	u64 rx_size_255;		/* prc255 */
	u64 rx_size_511;		/* prc511 */
	u64 rx_size_1023;		/* prc1023 */
	u64 rx_size_1522;		/* prc1522 */
	u64 rx_size_big;		/* prc9522 */
	u64 rx_undersize;		/* ruc */
	u64 rx_fragments;		/* rfc */
	u64 rx_oversize;		/* roc */
	u64 rx_jabber;			/* rjc */
	u64 tx_size_64;			/* ptc64 */
	u64 tx_size_127;		/* ptc127 */
	u64 tx_size_255;		/* ptc255 */
	u64 tx_size_511;		/* ptc511 */
	u64 tx_size_1023;		/* ptc1023 */
	u64 tx_size_1522;		/* ptc1522 */
	u64 tx_size_big;		/* ptc9522 */
	u64 mac_short_pkt_dropped;	/* mspdc */
#ifdef SDK_SUPPORT
	/* Counter specific to checksum_error (CSR: XEC) is dropped from HW
	 * (XEC is not implemented, reads to that CSR shall always return zero)
	 */
#endif /* SDK_SUPPORT */
#ifndef NO_UNUSED_CODE
	/* EEE LPI */
	u32 tx_lpi_status;
	u32 rx_lpi_status;
	u64 tx_lpi_count;		/* etlpic */
	u64 rx_lpi_count;		/* erlpic */
#endif /* !NO_UNUSED_CODE */
#ifdef FDIR_SUPPORT
	/* flow director stats */
#ifdef ATR_SUPPORT
	u64 fd_atr_match;
	u64 fd_atr_tunnel_match;
	u32 fd_atr_status;
#endif /* ATR_SUPPORT */
	u32 fd_sb_status;
	u64 fd_sb_match;
#ifdef ADQ_SUPPORT
	u64 ch_atr_match;
#endif /* ADQ_SUPPORT */
#ifdef ARFS_SUPPORT
#ifdef ICE_ADD_PROBES
	u64 arfs_tcpv4_match;
	u64 arfs_tcpv6_match;
	u64 arfs_udpv4_match;
	u64 arfs_udpv6_match;
#endif /* ICE_ADD_PROBES */
#endif /* ARFS_SUPPORT */
#endif /* FDIR_SUPPORT */
};

enum ice_sw_fwd_act_type {
	ICE_FWD_TO_VSI = 0,
	ICE_FWD_TO_VSI_LIST, /* Do not use this when adding filter */
	ICE_FWD_TO_Q,
	ICE_FWD_TO_QGRP,
	ICE_DROP_PACKET,
	ICE_INVAL_ACT
};

struct ice_aq_get_set_rss_lut_params {
	u16 vsi_handle;		/* software VSI handle */
	u16 lut_size;		/* size of the LUT buffer */
	u8 lut_type;		/* type of the LUT (i.e. VSI, PF, Global) */
	u8 *lut;		/* input RSS LUT for set and output RSS LUT for get */
	u8 global_lut_id;	/* only valid when lut_type is global */
};

/* Checksum and Shadow RAM pointers */
#define ICE_SR_NVM_CTRL_WORD			0x00
#define ICE_SR_PHY_ANALOG_PTR			0x04
#define ICE_SR_OPTION_ROM_PTR			0x05
#define ICE_SR_RO_PCIR_REGS_AUTO_LOAD_PTR	0x06
#define ICE_SR_AUTO_GENERATED_POINTERS_PTR	0x07
#define ICE_SR_PCIR_REGS_AUTO_LOAD_PTR		0x08
#define ICE_SR_EMP_GLOBAL_MODULE_PTR		0x09
#define ICE_SR_EMP_IMAGE_PTR			0x0B
#define ICE_SR_PE_IMAGE_PTR			0x0C
#define ICE_SR_CSR_PROTECTED_LIST_PTR		0x0D
#define ICE_SR_MNG_CFG_PTR			0x0E
#define ICE_SR_EMP_MODULE_PTR			0x0F
#define ICE_SR_PBA_BLOCK_PTR			0x16
#define ICE_SR_BOOT_CFG_PTR			0x132
#define ICE_SR_NVM_WOL_CFG			0x19
#define ICE_NVM_OROM_VER_OFF			0x02
#define ICE_SR_NVM_DEV_STARTER_VER		0x18
#define ICE_SR_ALTERNATE_SAN_MAC_ADDR_PTR	0x27
#define ICE_SR_PERMANENT_SAN_MAC_ADDR_PTR	0x28
#define ICE_SR_NVM_MAP_VER			0x29
#define ICE_SR_NVM_IMAGE_VER			0x2A
#define ICE_SR_NVM_STRUCTURE_VER		0x2B
#define ICE_SR_NVM_EETRACK_LO			0x2D
#define ICE_SR_NVM_EETRACK_HI			0x2E
#define ICE_NVM_VER_LO_SHIFT			0
#define ICE_NVM_VER_LO_MASK			(0xff << ICE_NVM_VER_LO_SHIFT)
#define ICE_NVM_VER_HI_SHIFT			12
#define ICE_NVM_VER_HI_MASK			(0xf << ICE_NVM_VER_HI_SHIFT)
#define ICE_OEM_EETRACK_ID			0xffffffff
#define ICE_OROM_VER_PATCH_SHIFT		0
#define ICE_OROM_VER_PATCH_MASK		(0xff << ICE_OROM_VER_PATCH_SHIFT)
#define ICE_OROM_VER_BUILD_SHIFT		8
#define ICE_OROM_VER_BUILD_MASK		(0xffff << ICE_OROM_VER_BUILD_SHIFT)
#define ICE_OROM_VER_SHIFT			24
#define ICE_OROM_VER_MASK			(0xff << ICE_OROM_VER_SHIFT)
#define ICE_SR_VPD_PTR				0x2F
#define ICE_SR_PXE_SETUP_PTR			0x30
#define ICE_SR_PXE_CFG_CUST_OPTIONS_PTR		0x31
#ifdef LEGACY_PREBOOT_SUPPORT
#define ICE_SR_IBA_CAPS_PTR			0x33
#endif /* LEGACY_PREBOOT_SUPPORT */
#define ICE_SR_NVM_ORIGINAL_EETRACK_LO		0x34
#define ICE_SR_NVM_ORIGINAL_EETRACK_HI		0x35
#define ICE_SR_VLAN_CFG_PTR			0x37
#define ICE_SR_POR_REGS_AUTO_LOAD_PTR		0x38
#define ICE_SR_EMPR_REGS_AUTO_LOAD_PTR		0x3A
#define ICE_SR_GLOBR_REGS_AUTO_LOAD_PTR		0x3B
#define ICE_SR_CORER_REGS_AUTO_LOAD_PTR		0x3C
#define ICE_SR_PHY_CFG_SCRIPT_PTR		0x3D
#define ICE_SR_PCIE_ALT_AUTO_LOAD_PTR		0x3E
#define ICE_SR_SW_CHECKSUM_WORD			0x3F
#define ICE_SR_PFA_PTR				0x40
#define ICE_SR_1ST_SCRATCH_PAD_PTR		0x41
#define ICE_SR_1ST_NVM_BANK_PTR			0x42
#define ICE_SR_NVM_BANK_SIZE			0x43
#define ICE_SR_1ST_OROM_BANK_PTR		0x44
#define ICE_SR_OROM_BANK_SIZE			0x45
#define ICE_SR_NETLIST_BANK_PTR			0x46
#define ICE_SR_NETLIST_BANK_SIZE		0x47
#define ICE_SR_EMP_SR_SETTINGS_PTR		0x48
#define ICE_SR_CONFIGURATION_METADATA_PTR	0x4D
#define ICE_SR_IMMEDIATE_VALUES_PTR		0x4E
#define ICE_SR_LINK_DEFAULT_OVERRIDE_PTR	0x134
#ifdef QV_SUPPORT
#define ICE_SR_FACTORY_SETTINGS_SECTION_POINTER		0x61
#endif /* QV_SUPPORT */
#define ICE_SR_POR_REGISTERS_AUTOLOAD_PTR	0x118

/* CSS Header words */
#define ICE_NVM_CSS_SREV_L			0x14
#define ICE_NVM_CSS_SREV_H			0x15

/* Length of CSS header section in words */
#define ICE_CSS_HEADER_LENGTH			330

/* Offset of Shadow RAM copy in the NVM bank area. */
#define ICE_NVM_SR_COPY_WORD_OFFSET		ROUND_UP(ICE_CSS_HEADER_LENGTH, 32)

/* Size in bytes of Option ROM trailer */
#define ICE_NVM_OROM_TRAILER_LENGTH		(2 * ICE_CSS_HEADER_LENGTH)

/* The Link Topology Netlist section is stored as a series of words. It is
 * stored in the NVM as a TLV, with the first two words containing the type
 * and length.
 */
#define ICE_NETLIST_LINK_TOPO_MOD_ID		0x011B
#define ICE_NETLIST_TYPE_OFFSET			0x0000
#define ICE_NETLIST_LEN_OFFSET			0x0001

/* The Link Topology section follows the TLV header. When reading the netlist
 * using ice_read_netlist_module, we need to account for the 2-word TLV
 * header.
 */
#define ICE_NETLIST_LINK_TOPO_OFFSET(n)		((n) + 2)

#define ICE_LINK_TOPO_MODULE_LEN		ICE_NETLIST_LINK_TOPO_OFFSET(0x0000)
#define ICE_LINK_TOPO_NODE_COUNT		ICE_NETLIST_LINK_TOPO_OFFSET(0x0001)

#define ICE_LINK_TOPO_NODE_COUNT_M		MAKEMASK(0x3FF, 0)

/* The Netlist ID Block is located after all of the Link Topology nodes. */
#define ICE_NETLIST_ID_BLK_SIZE			0x30
#define ICE_NETLIST_ID_BLK_OFFSET(n)		ICE_NETLIST_LINK_TOPO_OFFSET(0x0004 + 2 * (n))

/* netlist ID block field offsets (word offsets) */
#define ICE_NETLIST_ID_BLK_MAJOR_VER_LOW	0x02
#define ICE_NETLIST_ID_BLK_MAJOR_VER_HIGH	0x03
#define ICE_NETLIST_ID_BLK_MINOR_VER_LOW	0x04
#define ICE_NETLIST_ID_BLK_MINOR_VER_HIGH	0x05
#define ICE_NETLIST_ID_BLK_TYPE_LOW		0x06
#define ICE_NETLIST_ID_BLK_TYPE_HIGH		0x07
#define ICE_NETLIST_ID_BLK_REV_LOW		0x08
#define ICE_NETLIST_ID_BLK_REV_HIGH		0x09
#define ICE_NETLIST_ID_BLK_SHA_HASH_WORD(n)	(0x0A + (n))
#define ICE_NETLIST_ID_BLK_CUST_VER		0x2F

/* Auxiliary field, mask and shift definition for Shadow RAM and NVM Flash */
#define ICE_SR_VPD_SIZE_WORDS		512
#define ICE_SR_PCIE_ALT_SIZE_WORDS	512
#define ICE_SR_CTRL_WORD_1_S		0x06
#define ICE_SR_CTRL_WORD_1_M		(0x03 << ICE_SR_CTRL_WORD_1_S)
#define ICE_SR_CTRL_WORD_VALID		0x1
#define ICE_SR_CTRL_WORD_OROM_BANK	BIT(3)
#define ICE_SR_CTRL_WORD_NETLIST_BANK	BIT(4)
#define ICE_SR_CTRL_WORD_NVM_BANK	BIT(5)

#define ICE_SR_NVM_PTR_4KB_UNITS	BIT(15)

/* Shadow RAM related */
#define ICE_SR_SECTOR_SIZE_IN_WORDS	0x800
#define ICE_SR_BUF_ALIGNMENT		4096
#define ICE_SR_WORDS_IN_1KB		512
/* Checksum should be calculated such that after adding all the words,
 * including the checksum word itself, the sum should be 0xBABA.
 */
#define ICE_SR_SW_CHECKSUM_BASE		0xBABA

/* Link override related */
#define ICE_SR_PFA_LINK_OVERRIDE_WORDS		10
#define ICE_SR_PFA_LINK_OVERRIDE_PHY_WORDS	4
#define ICE_SR_PFA_LINK_OVERRIDE_OFFSET		2
#define ICE_SR_PFA_LINK_OVERRIDE_FEC_OFFSET	1
#define ICE_SR_PFA_LINK_OVERRIDE_PHY_OFFSET	2
#define ICE_FW_API_LINK_OVERRIDE_MAJ		1
#define ICE_FW_API_LINK_OVERRIDE_MIN		5
#define ICE_FW_API_LINK_OVERRIDE_PATCH		2

#define ICE_PBA_FLAG_DFLT		0xFAFA
/* Hash redirection LUT for VSI - maximum array size */
#define ICE_VSIQF_HLUT_ARRAY_SIZE	((VSIQF_HLUT_MAX_INDEX + 1) * 4)

/*
 * Defines for values in the VF_PE_DB_SIZE bits in the GLPCI_LBARCTRL register.
 * This is needed to determine the BAR0 space for the VFs
 */
#define GLPCI_LBARCTRL_VF_PE_DB_SIZE_0KB 0x0
#define GLPCI_LBARCTRL_VF_PE_DB_SIZE_8KB 0x1
#define GLPCI_LBARCTRL_VF_PE_DB_SIZE_64KB 0x2

#ifndef NO_DCB_SUPPORT
/* AQ API version for LLDP_FILTER_CONTROL */
#define ICE_FW_API_LLDP_FLTR_MAJ	1
#define ICE_FW_API_LLDP_FLTR_MIN	7
#define ICE_FW_API_LLDP_FLTR_PATCH	1
#endif /* !NO_DCB_SUPPORT */

/* AQ API version for report default configuration */
#define ICE_FW_API_REPORT_DFLT_CFG_MAJ		1
#define ICE_FW_API_REPORT_DFLT_CFG_MIN		7
#define ICE_FW_API_REPORT_DFLT_CFG_PATCH	3
#ifdef HEALTH_STATUS_SUPPORT

/* AQ API version for FW health reports */
#define ICE_FW_API_HEALTH_REPORT_MAJ		1
#define ICE_FW_API_HEALTH_REPORT_MIN		7
#define ICE_FW_API_HEALTH_REPORT_PATCH		6
#endif /* HEALTH_STATUS_SUPPORT */
#endif /* _ICE_TYPE_H_ */
