/* SPDX-License-Identifier: GPL-2.0 or Linux-OpenIB */
/* Copyright (c) 2021 Intel Corporation */
/*$FreeBSD$*/
#include "osdep.h"
#include "ice_rdma.h"
#include "main.h"
#include "irdma_di_if.h"
#include "fbsd_kcompat.h"
#if __FreeBSD_version >= 1300000
#include <sys/gsb_crc32.h>
#include <netinet/in_fib.h>
#include <netinet6/in6_fib.h>
#include <net/route/nhop.h>
#endif

bool		irdma_upload_context = FALSE;

inline u32
irdma_rd32(struct irdma_dev_ctx *dev_ctx, u32 reg){

	KASSERT(reg < dev_ctx->mem_bus_space_size,
		("iw_ixl: register offset %#jx too large (max is %#jx)",
		 (uintmax_t)reg, (uintmax_t)dev_ctx->mem_bus_space_size));

	return (bus_space_read_4(dev_ctx->mem_bus_space_tag,
				 dev_ctx->mem_bus_space_handle, reg));
}

inline void
irdma_wr32(struct irdma_dev_ctx *dev_ctx, u32 reg, u32 value)
{

	KASSERT(reg < dev_ctx->mem_bus_space_size,
		("iw_ixl: register offset %#jx too large (max is %#jx)",
		 (uintmax_t)reg, (uintmax_t)dev_ctx->mem_bus_space_size));

	bus_space_write_4(dev_ctx->mem_bus_space_tag,
			  dev_ctx->mem_bus_space_handle, reg, value);
}

inline u64
irdma_rd64(struct irdma_dev_ctx *dev_ctx, u32 reg){

	KASSERT(reg < dev_ctx->mem_bus_space_size,
		("iw_ixl: register offset %#jx too large (max is %#jx)",
		 (uintmax_t)reg, (uintmax_t)dev_ctx->mem_bus_space_size));

	return (bus_space_read_8(dev_ctx->mem_bus_space_tag,
				 dev_ctx->mem_bus_space_handle, reg));
}

inline void
irdma_wr64(struct irdma_dev_ctx *dev_ctx, u32 reg, u64 value)
{

	KASSERT(reg < dev_ctx->mem_bus_space_size,
		("iw_ixl: register offset %#jx too large (max is %#jx)",
		 (uintmax_t)reg, (uintmax_t)dev_ctx->mem_bus_space_size));

	bus_space_write_8(dev_ctx->mem_bus_space_tag,
			  dev_ctx->mem_bus_space_handle, reg, value);

}

enum irdma_status_code
irdma_register_qset(struct irdma_sc_vsi *vsi, struct irdma_ws_node *tc_node)
{
	struct irdma_device *iwdev = vsi->back_vsi;
	struct ice_rdma_peer *peer = iwdev->rf->pdev;
	struct ice_rdma_qset_update rdma_qset_res = {0};
	int		ret;

	rdma_qset_res.cnt_req = 1;
	rdma_qset_res.res_type = ICE_RDMA_QSET_ALLOC;
	rdma_qset_res.qsets.qs_handle = tc_node->qs_handle;
	rdma_qset_res.qsets.tc = tc_node->traffic_class;
	rdma_qset_res.qsets.vsi_id = vsi->vsi_idx;
	ret = IRDMA_DI_QSET_REGISTER_REQUEST(peer, &rdma_qset_res);
	if (ret) {
		irdma_debug(vsi->dev, IRDMA_DEBUG_WS,
			    "LAN alloc_res for rdma qset failed.\n");
		return IRDMA_ERR_NO_MEMORY;
	}

	tc_node->l2_sched_node_id = rdma_qset_res.qsets.teid;
	vsi->qos[tc_node->user_pri].l2_sched_node_id =
	    rdma_qset_res.qsets.teid;

	return 0;
}
void
irdma_unregister_qset(struct irdma_sc_vsi *vsi, struct irdma_ws_node *tc_node)
{
	struct irdma_device *iwdev = vsi->back_vsi;
	struct ice_rdma_peer *peer = iwdev->rf->pdev;
	struct ice_rdma_qset_update rdma_qset_res = {0};

	rdma_qset_res.res_allocated = 1;
	rdma_qset_res.res_type = ICE_RDMA_QSET_FREE;
	rdma_qset_res.qsets.vsi_id = vsi->vsi_idx;
	rdma_qset_res.qsets.teid = tc_node->l2_sched_node_id;
	rdma_qset_res.qsets.qs_handle = tc_node->qs_handle;

	if (IRDMA_DI_QSET_REGISTER_REQUEST(peer, &rdma_qset_res))
		irdma_debug(vsi->dev, IRDMA_DEBUG_WS,
			    "LAN free_res for rdma qset failed.\n");
}

struct device  *
hw_to_dev(struct irdma_hw *hw)
{
	return ((struct irdma_dev_ctx *)hw->dev_context)->dev;
}

void
irdma_free_hash_desc(void *desc)
{
	return;
}

enum irdma_status_code
irdma_init_hash_desc(void **desc)
{
	return IRDMA_SUCCESS;
}

enum irdma_status_code
irdma_ieq_check_mpacrc(void *desc,
		       void *addr, u32 len, u32 val)
{
	u32		crc = calculate_crc32c(0xffffffff, addr, len) ^ 0xffffffff;
	enum irdma_status_code ret_code = IRDMA_SUCCESS;

	if (crc != val) {
		irdma_pr_err("mpa crc check fail %x %x\n", crc, val);
		ret_code = IRDMA_ERR_MPA_CRC;
	}
	printf("%s: result crc=%x value=%x\n", __func__, crc, val);
	return ret_code;
}

/**
 * irdma_add_ipv6_addr - add ipv6 address to the hw arp table
 * @iwdev: irdma device
 */
void
irdma_add_ipv6_addr(struct irdma_device *iwdev)
{
	struct ifnet   *ifp = iwdev->rf->netdev;
	struct ifaddr  *ifa,
		       *tmp;
	struct sockaddr_in6 *sin6;
	u32		local_ipaddr6[4];
	u8	       *mac_addr;
	char		ip6buf[INET6_ADDRSTRLEN];

	if_addr_rlock(ifp);
	IRDMA_TAILQ_FOREACH_SAFE(ifa, &ifp->if_addrhead, ifa_link, tmp) {
		sin6 = (struct sockaddr_in6 *)ifa->ifa_addr;
		if (sin6->sin6_family != AF_INET6)
			continue;

		irdma_copy_ip_ntohl(local_ipaddr6, (u32 *)&sin6->sin6_addr);
		mac_addr = IF_LLADDR(ifp);

		printf("%s:%d IP=%s, MAC=%02x:%02x:%02x:%02x:%02x:%02x\n",
		       __func__, __LINE__,
		       ip6_sprintf(ip6buf, &sin6->sin6_addr),
		       mac_addr[0], mac_addr[1], mac_addr[2],
		       mac_addr[3], mac_addr[4], mac_addr[5]);

		irdma_manage_arp_cache(iwdev->rf, mac_addr, local_ipaddr6, FALSE,
				       IRDMA_ARP_ADD);

	}
	if_addr_runlock(ifp);
}

/**
 * irdma_add_ipv4_addr - add ipv4 address to the hw arp table
 * @iwdev: irdma device
 */
void
irdma_add_ipv4_addr(struct irdma_device *iwdev)
{
	struct ifnet   *ifp = iwdev->rf->netdev;
	struct ifaddr  *ifa;
	struct sockaddr_in *sin;
	in_addr_t	ip_addr;
	u8	       *mac_addr;

	if_addr_rlock(ifp);
	IRDMA_TAILQ_FOREACH(ifa, &ifp->if_addrhead, ifa_link) {
		sin = (struct sockaddr_in *)ifa->ifa_addr;
		if (sin->sin_family != AF_INET)
			continue;

		ip_addr = ntohl(sin->sin_addr.s_addr);
		mac_addr = IF_LLADDR(ifp);

		printf("%s:%d IP=%d.%d.%d.%d, MAC=%02x:%02x:%02x:%02x:%02x:%02x\n",
		       __func__, __LINE__,
		       ip_addr >> 24,
		       (ip_addr >> 16) & 0xFF,
		       (ip_addr >> 8) & 0xFF,
		       ip_addr & 0xFF,
		       mac_addr[0], mac_addr[1], mac_addr[2],
		       mac_addr[3], mac_addr[4], mac_addr[5]);

		irdma_manage_arp_cache(iwdev->rf, mac_addr, &ip_addr, TRUE,
				       IRDMA_ARP_ADD);
	}
	if_addr_runlock(ifp);
}

/**
 * irdma_add_ip - add ip addresses
 * @iwdev: irdma device
 *
 * Add ipv4/ipv6 addresses to the arp cache
 */
void
irdma_add_ip(struct irdma_device *iwdev)
{
	irdma_add_ipv4_addr(iwdev);
	irdma_add_ipv6_addr(iwdev);
}

static int
irdma_get_route_ifp(struct sockaddr *dst_sin, struct net_device *netdev,
		    struct ifnet **ifp, struct sockaddr **nexthop, bool *gateway)
{
#if __FreeBSD_version >= 1300000
	struct nhop_object *nh;

	if (dst_sin->sa_family == AF_INET6)
		nh = fib6_lookup(RT_DEFAULT_FIB, &((struct sockaddr_in6 *)dst_sin)->sin6_addr, 0, NHR_NONE, 0);
	else
		nh = fib4_lookup(RT_DEFAULT_FIB, ((struct sockaddr_in *)dst_sin)->sin_addr, 0, NHR_NONE, 0);
	if (!nh || (nh->nh_ifp != netdev &&
		    rdma_vlan_dev_real_dev(nh->nh_ifp) != netdev))
		goto rt_not_found;
	*gateway = (nh->nh_flags & NHF_GATEWAY) ? true : false;
	*nexthop = (*gateway) ? &nh->gw_sa : dst_sin;
	*ifp = nh->nh_ifp;
#else
	struct rtentry *rte;

	rte = rtalloc1(dst_sin, 0, 0);
	if (!rte || (rte->rt_ifp != netdev &&
		     rdma_vlan_dev_real_dev(rte->rt_ifp) != netdev)) {
		if (rte)
			RTFREE_LOCKED(rte);
		goto rt_not_found;
	}
	*gateway = (rte->rt_flags & RTF_GATEWAY) ? true : false;
	*nexthop = (*gateway) ? rte->rt_gateway :
	    dst_sin;
	*ifp = rte->rt_ifp;

	RTFREE_LOCKED(rte);
#endif

	return 0;

rt_not_found:
	pr_err("irdma: route not found\n");
	return -ENETUNREACH;
}

/**
 * irdma_get_dst_mac - get destination mac address
 * @cm_node: connection's node
 * @dst_sin: destination address information
 * @dst_mac: mac address array to return
 */
int
irdma_get_dst_mac(struct irdma_cm_node *cm_node, struct sockaddr *dst_sin, u8 *dst_mac)
{
	struct net_device *netdev = cm_node->iwdev->netdev;
#ifdef VIMAGE
	struct rdma_cm_id *rdma_id = (struct rdma_cm_id *)cm_node->cm_id->context;
	struct vnet    *vnet = rdma_id->route.addr.dev_addr.net;
#endif
	struct ifnet   *ifp;
	struct llentry *lle;
	struct sockaddr *nexthop;
	int		err;
	bool		gateway;

#ifdef VIMAGE
	CURVNET_SET_QUIET(vnet);
#endif
	err = irdma_get_route_ifp(dst_sin, netdev, &ifp, &nexthop, &gateway);
	if (err)
		goto get_route_fail;

	if (dst_sin->sa_family == AF_INET) {
		err = arpresolve(ifp, gateway, NULL, nexthop, dst_mac, NULL, &lle);
	} else if (dst_sin->sa_family == AF_INET6) {
		err = nd6_resolve(ifp, gateway, NULL, nexthop, dst_mac, NULL, &lle);
	} else {
		err = -EPROTONOSUPPORT;
	}

get_route_fail:
#ifdef VIMAGE
	CURVNET_RESTORE();
#endif
	if (err) {
		pr_err("failed to resolve neighbor address (err=%d)\n",
		       err);
		return -ENETUNREACH;
	}

	return 0;
}

/**
 * irdma_addr_resolve_neigh - resolve neighbor address
 * @cm_node: connection's node
 * @dst_ip: remote ip address
 * @arpindex: if there is an arp entry
 */
int
irdma_addr_resolve_neigh(struct irdma_cm_node *cm_node,
			 u32 dst_ip, int arpindex)
{
	struct irdma_device *iwdev = cm_node->iwdev;
	struct sockaddr_in dst_sin = {};
	int		err;
	u8		dst_mac[MAX_ADDR_LEN];

	dst_sin.sin_len = sizeof(dst_sin);
	dst_sin.sin_family = AF_INET;
	dst_sin.sin_port = 0;
	dst_sin.sin_addr.s_addr = htonl(dst_ip);

	err = irdma_get_dst_mac(cm_node, (struct sockaddr *)&dst_sin, dst_mac);
	if (err)
		return arpindex;

	return irdma_add_arp(iwdev->rf, &dst_ip, true, dst_mac);
}

/**
 * irdma_addr_resolve_neigh_ipv6 - resolve neighbor ipv6 address
 * @cm_node: connection's node
 * @dest: remote ip address
 * @arpindex: if there is an arp entry
 */
int
irdma_addr_resolve_neigh_ipv6(struct irdma_cm_node *cm_node,
			      u32 *dest, int arpindex)
{
	struct irdma_device *iwdev = cm_node->iwdev;
	struct sockaddr_in6 dst_addr = {};
	int		err;
	u8		dst_mac[MAX_ADDR_LEN];

	dst_addr.sin6_family = AF_INET6;
	dst_addr.sin6_len = sizeof(dst_addr);
	dst_addr.sin6_scope_id = iwdev->netdev->if_index;

	irdma_copy_ip_htonl(dst_addr.sin6_addr.__u6_addr.__u6_addr32, dest);
	err = irdma_get_dst_mac(cm_node, (struct sockaddr *)&dst_addr, dst_mac);
	if (err)
		return arpindex;

	return irdma_add_arp(iwdev->rf, dest, false, dst_mac);
}

/**
 * irdma_add_handler - add a handler to the list
 * @hdl: handler to be added to the handler list
 */
void
irdma_add_handler(struct irdma_handler *hdl)
{
	unsigned long	flags;

	spin_lock_irqsave(&irdma_handler_lock, flags);
	list_add(&hdl->list, &irdma_handlers);
	spin_unlock_irqrestore(&irdma_handler_lock, flags);
}

/**
 * irdma_del_handler - delete a handler from the list
 * @hdl: handler to be deleted from the handler list
 */
void
irdma_del_handler(struct irdma_handler *hdl)
{
	unsigned long	flags;

	spin_lock_irqsave(&irdma_handler_lock, flags);
	list_del(&hdl->list);
	spin_unlock_irqrestore(&irdma_handler_lock, flags);
}

void
irdma_set_rf_user_cfg_params(struct irdma_pci_f *rf)
{
	int		en_rem_endpoint_trk = 0;
	int		limits_sel = 4;

	rf->en_rem_endpoint_trk = en_rem_endpoint_trk;
	rf->limits_sel = limits_sel;
	rf->rst_to = IRDMA_RST_TIMEOUT_HZ;
}

/**
 * irdma_dmamap_cb - callback for bus_dmamap_load
 *
 */
static void
irdma_dmamap_cb(void *arg, bus_dma_segment_t * segs, int nseg, int error)
{
	if (error)
		return;
	*(bus_addr_t *) arg = segs->ds_addr;
	return;
}

/**
 * irdma_allocate_dma_mem - allocate dma memory
 * @hw: pointer to hw structure
 * @mem: structure holding memory information
 * @size: requested size
 * @alignment: requested alignment
 */
void	       *
irdma_allocate_dma_mem(struct irdma_hw *hw, struct irdma_dma_mem *mem,
		       u64 size, u32 alignment)
{
	struct irdma_dev_ctx *dev_ctx = (struct irdma_dev_ctx *)hw->dev_context;
	device_t	dev = dev_ctx->dev;
	void	       *va;
	int		ret;

	ret = bus_dma_tag_create(bus_get_dma_tag(dev),	/* parent */
				 alignment, 0,	/* alignment, bounds */
				 BUS_SPACE_MAXADDR,	/* lowaddr */
				 BUS_SPACE_MAXADDR,	/* highaddr */
				 NULL, NULL,	/* filter, filterarg */
				 size,	/* maxsize */
				 1,	/* nsegments */
				 size,	/* maxsegsize */
				 BUS_DMA_ALLOCNOW,	/* flags */
				 NULL,	/* lockfunc */
				 NULL,	/* lockfuncarg */
				 &mem->tag);
	if (ret != 0) {
		device_printf(dev, "%s: bus_dma_tag_create failed, error %u\n",
			      __func__, ret);
		goto fail_0;
	}
	ret = bus_dmamem_alloc(mem->tag, (void **)&va,
			       BUS_DMA_NOWAIT | BUS_DMA_ZERO, &mem->map);
	if (ret != 0) {
		device_printf(dev, "%s: bus_dmamem_alloc failed, error %u\n",
			      __func__, ret);
		goto fail_1;
	}
	ret = bus_dmamap_load(mem->tag, mem->map, va, size,
			      irdma_dmamap_cb, &mem->pa, BUS_DMA_NOWAIT);
	if (ret != 0) {
		device_printf(dev, "%s: bus_dmamap_load failed, error %u\n",
			      __func__, ret);
		goto fail_2;
	}
	mem->nseg = 1;
	mem->size = size;
	bus_dmamap_sync(mem->tag, mem->map,
			BUS_DMASYNC_PREREAD | BUS_DMASYNC_PREWRITE);

	return va;
fail_2:
	bus_dmamem_free(mem->tag, va, mem->map);
fail_1:
	bus_dma_tag_destroy(mem->tag);
fail_0:
	mem->map = NULL;
	mem->tag = NULL;

	return NULL;
}

/**
 * irdma_free_dma_mem - Memory free helper fn
 * @hw: pointer to hw structure
 * @mem: ptr to mem struct to free
 */
enum irdma_status_code
irdma_free_dma_mem(struct irdma_hw *hw, struct irdma_dma_mem *mem)
{
	if (!mem)
		return IRDMA_ERR_PARAM;
	bus_dmamap_sync(mem->tag, mem->map,
			BUS_DMASYNC_POSTREAD | BUS_DMASYNC_POSTWRITE);
	bus_dmamap_unload(mem->tag, mem->map);
	if (!mem->va)
		return IRDMA_ERR_NO_MEMORY;
	bus_dmamem_free(mem->tag, mem->va, mem->map);
	bus_dma_tag_destroy(mem->tag);

	mem->va = NULL;

	return IRDMA_SUCCESS;
}

inline void
irdma_prm_rem_bitmapmem(struct irdma_hw *hw, struct irdma_chunk *chunk)
{
	kfree(chunk->bitmapmem.va);
}

u64
ether_addr_to_u64(u8 *addr)
{
	u64		mac = 0;
	int		i;

	for (i = 0; i < ETH_ALEN; ++i) {
		mac = mac << 8;
		mac |= addr[i];
	}
	return mac;
}
